# Copyright (C) 2004 Matthew Colyer
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

from xml.dom import minidom
import copy

class Operation(minidom.Element):
	"""An abstract class which all operations are derived from."""
	def getSite(self):
		return self.__site
	def setSite(self, site):
		#FIXME: Verify that it is an integer, else exception
		self.__site = site
		self.setAttribute("site", str(site))

class TextOperation(Operation):
	"""An abstract class which all text operations are derived from."""
	def getPosition(self):
		return self.__position
	def setPosition(self, position):
		#FIXME: Verify that it is an integer, else exception
		self.__position = position
		self.setAttribute("position", str(position))

class Cursor(TextOperation):
	"""An operation which represents a cursor movement."""
	def __init__(self, site, position):
		minidom.Element.__init__(self, "cursor")
		self.setSite(site)
		self.setPosition(position)
	def __str__(self):
		return "Cursor Pos="+str(self.getPosition())

class GroupOperation(TextOperation):
	"""An abstract class which all group operations are derived from."""
	def getState(self):
		return self.__state
	def setState(self, state):
		self.__state = state
		self.setAttribute("position", str(state))
	
	def isStable(self, current):
		site = self.getSite()
		state = self.getState()
		
		for i,clock in state.iteritems():
			if i < site:
				if clock >  current[i]:
					return False
			elif i >= site:
				if clock > current[i]+1:
					return False
		return True

	def isIndependentOf(self, operation):
		if ((not self.precedes(operation)) and (not operation.precedes(self))):
			return True
		return False
	
	def precedes(self, operation):
		operation_a_site= self.getSite()
		operation_b_site = operation.getSite()
		operation_a_state = self.getState()
		operation_b_state = operation.getState()
		
		if (operation_a_site == operation_b_site and
		    operation_a_state[operation_a_site] < operation_b_state[operation_b_site]):
			return True
		elif (operation_a_site != operation_b_site):
			for site_a,clock_a in operation_a_state.iteritems():
					if (clock_a > operation_b_state[site_a]):
						return False
			return True

	def compare(self, operation):
		operation_a_site= self.getSite()
		operation_b_site = operation.getSite()
		operation_a_state = self.getState()
		operation_b_state = operation.getState()
		
		a_clock_sum = 0
		b_clock_sum = 0
		
		for site,clock in operation_b_state.iteritems():
			a_clock_sum += operation_a_state[site]
			b_clock_sum += clock
		
		if (a_clock_sum < b_clock_sum):
			return 1
		elif (a_clock_sum > b_clock_sum):
			return -1
		else:
			if (operation_a_site > operation_b_site):
				return -1
			elif (operation_a_site < operation_b_site):
				return 1

	def inclusionTransformation(self, operation):
		if self.hasRelativeAddress():
			if self.checkBaseOperation(operation):
				executableOperations = [self.convertToAbsoluteAddress(operation)]
			else:
				executableOperations = [self]
		elif operation.tagName == "insert":
			executableOperations = self.inclusionAgainstInsert(operation)
		elif operation.tagName == "delete":
			executableOperations = self.inclusionAgainstDelete(operation)
		else:
			#FIXME: Throw exception
			print "ERROR"
		return executableOperations
	def exclusionTransformation(self, operation):
		if self.hasRelativeAddress():
			executableOperations = [self]
		elif operation.tagName == "insert":
			executableOperations = self.exclusionAgainstInsert(operation)
		elif operation.tagName == "delete":
			executableOperations = self.exclusionAgainstDelete(operation)
		else:
			print "ERROR"
		return executableOperations
	
	def listInclusionTransformationExt(self, listOfOperations):
		if len(listOfOperations) == 0:
			return OperationList([self])
		elif self.hasRelativeAddress() and not(self.checkBaseOperation(listOfOperations[0])):
			return self.listInclusionTransformationExt(listOfOperations[1:])
		elif self.hasRelativeAddress() and self.checkBaseOperation(listOfOperations[0]):
			self.convertToAbsoluteAddress(listOfOperations[0])
			return self.listInclusionTransformationExt(listOfOperations[1:])
		else:
			newList = OperationList(self.inclusionTransformation(listOfOperations[0]))
			return newList.listInclusionTransformation(listOfOperations[1:])
	def listExclusionTransformationExt(self, listOfOperations):
		if self.hasRelativeAddress() or len(listOfOperations) == 0:
			return OperationList([self])
		else:
			newList = OperationList(self.exclusionTransformation(listOfOperations[0]))
			return newList.listExclusionTransformation(listOfOperations[1:])


	#Transformation Helper Functions
	def saveRelativeAddress(self, operation):
		state = operation.getState()
		site = operation.getSite()
		index = str(site)+"-"+str(state[site])
		
		self._relative = index
	def hasRelativeAddress(self):
		try:
			if self._relative != None:
				return True
			else:
				return False
		except AttributeError:
			#If a relative address has not already been set then 
			#this exception will occur
			return False
	
	def checkBaseOperation(self, operation):
		state = operation.getState()
		site = operation.getSite()
		index = str(site)+"-"+str(state[site])
		
		return self._relative == index
	
	def convertToAbsoluteAddress(self, operation):
		self.setPosition(operation.getPosition()+self.getPosition())
		self._relative = None
		return self

class Insert(GroupOperation):
	"""A class which represents inserted text."""
	def __init__(self, site, state, string, position):
		minidom.Element.__init__(self, "insert")
		self.__backup = {}
		self.setSite(site)
		self.setState(state)
		self.setString(string)
		self.setPosition(position)
	
	def getString(self):
		return self.__string
	def setString(self, string):
		self.__string = string
		self.setAttribute("string", str(string))
	
	def getLength(self):
		return len(self.__string)
	
	def __str__(self):
		return "Insert pos="+str(self.getPosition())+" str="+self.getString()
	
	def apply(self, buffer):
		string = self.getString()
		position = self.getPosition()
		
		for i,character in enumerate(string):
			buffer.insert(position+i, character)
	def unapply(self, buffer):
		position = self.getPosition()
		length = self.getLength()
		
		for i in range(length):
			for j in range(position, len(buffer)-1):
				buffer[j] = buffer[j+1]
	
	def inclusionAgainstInsert(self, operation):
		o_a_pos = self.getPosition()
		o_b_pos = operation.getPosition()
		o_b_len = operation.getLength()
		
		#If it is to the left of the second operation, do nothing
		if (o_a_pos < o_b_pos):
			pass
		#If it is to the right or starts at the same place as the second
		#operation, shift it the proper amount
		else:
			self.setPosition(o_a_pos+o_b_len)
		
		return OperationList([self])

	def inclusionAgainstDelete(self, operation):
		o_a_pos = self.getPosition()
		o_b_pos = operation.getPosition()
		o_b_len = operation.getLength()
		
		#If it is to the left or starts at the same place as the delete
		#operation, do nothing
		if (o_a_pos <= o_b_pos):
			pass
		#If it is to the right of the delete operation, shift it by the proper amount
		elif (o_a_pos > o_b_pos+o_b_len):
			self.setPosition(o_a_pos-o_b_len)
		#If it is within the middle of the delete operation, shift it to the beginning
		#of the delete operation, this is arbitrary.
		else:
			self.saveLostInformation(operation)
			self.setPosition(o_b_pos)
		
		return OperationList([self])
	
	def exclusionAgainstInsert(self, operation):
		o_a_pos = self.getPosition()
		o_b_pos = operation.getPosition()
		o_b_len = operation.getLength()
				
		#If the string to transform is to the left of the one against
		if (o_a_pos <= o_b_pos):
			pass
		#If the string is to the right of the it is against
		elif (o_a_pos >= (o_b_pos + o_b_len)):
			self.setPosition(o_a_pos - o_b_len)
		#Otherwise it is in the middle
		else :
			self.setPosition(o_a_pos - o_b_pos)
			self.saveRelativeAddress(operation)
		
		return OperationList([self])
			
	def exclusionAgainstDelete(self, operation):
		o_a_pos = self.getPosition()
		o_b_pos = operation.getPosition()
		o_b_len = operation.getLength()
				
		if self.checkLostInformation(operation):
			self.recoverLostInformation(operation)
		
		#If the operation is to the left
		elif (o_a_pos <= o_b_pos):
			pass
		#Otherwise put it past the delete command
		else:
			self.setPosition(o_a_pos + o_b_len)
		
		return OperationList([self])
		
	
	def saveLostInformation(self, operation):
		state = operation.getState()
		site = operation.getSite()
		index = str(site)+"-"+str(state[site])
		
		self.__backup[index] = (self.getString(), self.getPosition())
	def checkLostInformation(self, operation):
		state = operation.getState()
		site = operation.getSite()
		index = str(site)+"-"+str(state[site])
		
		try:
			self.__backup[index]
			return True
		except KeyError:
			#There most likely will not be a value in the dictionary
			#for the variable and that means that there is no lost
			#information
			return False
	def recoverLostInformation(self, operation):
		state = operation.getState()
		site = operation.getSite()
		index = str(site)+"-"+str(state[site])
		
		self.setString(self.__backup[index][0])
		self.setPosition(self.__backup[index][1])

class Delete(GroupOperation):
	"""A class which represents deleted text."""
	def __init__(self, site, state, length, position):
		minidom.Element.__init__(self, "delete")
		self.__backup = {}
		self.__backupString = []
		self.setSite(site)
		self.setState(state)
		self.setLength(length)
		self.setPosition(position)
	
	def setLength(self, length):
		#FIXME: Verify that it is an integer, else exception
		self.__length = length
		self.setAttribute("length", str(length))
	def getLength(self):
		return self.__length
	
	def __str__(self):
		return "Delete pos="+str(self.getPosition())+" len="+str(self.getLength())
	
	def apply(self, buffer):
		length = self.getLength()
		position = self.getPosition()
		
		for i in range(length):
			self.__backupString.append(buffer[position])
			buffer.remove(buffer[position])
	def unapply(self, buffer):
		position = self.getPosition()
		length = self.getLength()
		
		for i in range(length):
			buffer[position+i] = self.__backupString[i]
	
	def inclusionAgainstInsert(self, operation):
		o_a_pos = self.getPosition()
		o_b_pos = operation.getPosition()
		o_a_len = self.getLength()
		o_b_len = operation.getLength()
		
		#If it is before or the starts at the same place as the insert
		#operation, do nothing
		if (o_b_pos >= (o_a_pos + o_a_len)):
			pass
		#If it is after or at the same position as the insert operation, shift
		#it by the proper amount. Arbitrarily puts the delete after the
		#insert.
		elif (o_a_pos >= o_b_pos):
			self.setPosition(o_a_pos+o_b_len)
		#Otherwise the delete contains the insert and two delete commands are required
		else:
			self.setLength(o_b_pos-o_a_pos)
			additionalDelete = copy.copy(self)
			additionalDelete.setLength(o_a_len-(o_b_pos-o_a_pos))
			additionalDelete.setPosition(o_b_pos+o_b_len)
			
			return OperationList([self, additionalDelete])
		
		return OperationList([self])
	def inclusionAgainstDelete(self, operation):
		o_a_pos = self.getPosition()
		o_b_pos = operation.getPosition()
		o_a_len = self.getLength()
		o_b_len = operation.getLength()
		
		#If it is to the left of the delete, do nothing
		if (o_b_pos >= (o_a_pos + o_a_len)):
			pass
		#If it is to the right of the delete, shift it by the proper amount
		elif (o_a_pos >= (o_b_pos + o_b_len)):
			self.setPosition(o_a_pos-o_b_len)
			print operation
		#If the first delete contains any portion of the second delete
		else:
			self.saveLostInformation(operation)
			
			#The first delete is contained completely within the second
			if (o_b_pos <= o_a_pos and ((o_a_pos + o_a_len) <= (o_b_pos + o_b_len))):
				self.setLength(0)
			
			#If it overlaps to the right of the second delete
			elif (o_b_pos <= o_a_pos and ((o_a_pos + o_a_len) > (o_b_pos + o_b_len))):
				self.setPosition(o_b_pos)
				self.setLength(o_a_pos+o_a_len-(o_b_pos+o_b_len))
				
			#If it overlaps to the left of the second delete
			elif (o_b_pos > o_a_pos and ((o_b_pos + o_b_len) >= (o_a_pos + o_a_len))):
				self.setLength(o_b_pos-o_a_pos)
			#If both start at same position and op_a is longer than op_b
			#resize it.
			else:
				self.setLength(o_a_len-o_b_len)
		return OperationList([self])
	
	def exclusionAgainstInsert(self, operation):
		o_a_pos = self.getPosition()
		o_b_pos = operation.getPosition()
		o_a_len = self.getLength()
		o_b_len = operation.getLength()
		
		if ((o_a_pos + o_a_len) <= o_b_pos):
			pass
		elif (o_a_pos >= (o_b_pos+o_b_len)):
			self.setPosition(o_a_pos - o_b_len)
		else:
			#Delete contained within the insert
			if (o_b_pos <= o_a_pos and  (o_a_pos + o_a_len) <= (o_b_pos + o_b_len)):
				self.setPosition(o_a_pos - o_b_pos)
			#Delete starts inside the insert and goes past the end of the insert
			elif (o_b_pos <= o_a_pos and (o_a_pos + o_a_len) > (o_b_pos + o_b_len)):
				self.setLength(o_b_pos+o_b_len-o_a_pos)
				self.setPosition(o_a_pos-o_b_pos)
				additionalDelete = copy.copy(self)
				additionalDelete.setLength((o_a_pos+o_a_len)-(o_b_pos+o_b_len))
				additionalDelete.setPosition(o_b_pos)
				
				self.saveRelativeAddress(operation)
				return OperationList([self, additionalDelete])
			#Delete encloses the entire insert
			elif (o_a_pos < o_b_pos and (o_b_pos + o_b_len) <= (o_a_pos+o_a_len)): 
				self.setLength(o_b_len)
				self.setPosition(0)
				additionalDelete = copy.copy(self)
				additionalDelete.setLength(o_a_len-o_b_len)
				additionalDelete.setPosition(o_a_pos)
				
				self.saveRelativeAddress(operation)
				return OperationList([self, additionalDelete])
			#Delete starts before the insert and ends inside it
			else:
				self.setLength(o_a_pos+o_a_len-o_b_pos)
				self.setPosition(0)
				additionalDelete = copy.copy(self)
				additionalDelete.setLength(o_b_pos-o_a_pos)
				additionalDelete.setPosition(o_a_pos)
				
				self.saveRelativeAddress(operation)
				return OperationList([self, additionalDelete])
			self.saveRelativeAddress(operation)
		return OperationList([self])
	def exclusionAgainstDelete(self, operation):
		o_a_pos = self.getPosition()
		o_b_pos = operation.getPosition()
		o_a_len = self.getLength()
		o_b_len = operation.getLength()
		
		if (self.checkLostInformation(operation)):
			self.recoverLostInformation(operation)
		elif (o_b_pos >= (o_a_pos + o_a_len)):
			pass
		elif (o_a_pos >= o_b_pos):
			self.setPosition(o_a_pos + o_b_len)
		else:
			self.setLength(o_b_pos-o_a_pos)
			additionalDelete = copy.copy(self)
			additionalDelete.setLength(o_a_len-(o_b_pos-o_a_pos))
			additionalDelete.setPosition(o_b_pos+o_b_len)
			
			return OperationList([self, additionalDelete])
		return OperationList([self])
	
	def saveLostInformation(self, operation):
		state = operation.getState()
		site = operation.getSite()
		index = str(site)+"-"+str(state[site])
		
		self.__backup[index] = (self.getLength(), self.getPosition())
	def checkLostInformation(self, operation):
		state = operation.getState()
		site = operation.getSite()
		index = str(site)+"-"+str(state[site])
		
		try:
			self.__backup[index]
			return True
		except KeyError:
			#There most likely will not be a value in the dictionary
			#for the variable and that means that there is no lost
			#information
			return False
	def recoverLostInformation(self, operation):
		state = operation.getState()
		site = operation.getSite()
		index = str(site)+"-"+str(state[site])
		
		self.setLength(self.__backup[index][0])
		self.setPosition(self.__backup[index][1])

class OperationList(list):
	"""A list specially designed to handle group operations"""
	def copy(self):
		operationsPrime = OperationList()
		for operation in self:
			operationsPrime.append(copy.copy(operation))
		return operationsPrime
	
	def listInclusionTransformation(self, listOfOperations):
		if len(self) == 0:
			return OperationList()
		else:
			tempListOfOperations1 = self[0].listInclusionTransformationExt(listOfOperations)
			tempListOfOperations2 = self[1:].listInclusionTransformation(listOfOperations + tempListOfOperations1)
			return tempListOfOperations1 + tempListOfOperations2
	
	def listExclusionTransformation(self, listOfOperations2):
		if len(self) == 0:
			return []
		else:
			tempListOfOperations1 = self[0].listExclusionTransformationExt(listOfOperations2)
			tempListOfOperations2 = self[1:].listExclusionTransformation(listOfOperations2 + OperationList([tempListOfOperations1]))
			return tempListOfOperations1 + tempListOfOperations2
	
	def __str__(self):
		result = "["
		for operation in self:
			result = result + str(operation) + " "
		return result[:-1]+"]"
	
	def __getslice__(self,x,y):
		return OperationList(list.__getslice__(self,x,y))
	def __add__(x,y):
		return OperationList(list.__add__(x,y))
