/*
 * Copyright (C) 2005 Robert Staudinger <robsta@stereolyzer.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef ABI_PLUGIN_BUILTIN
#define abi_plugin_register abipgn_userpluginloader_register
#define abi_plugin_unregister abipgn_userpluginloader_unregister
#define abi_plugin_supports_version abipgn_userpluginloader_supports_version
#endif

#include "ut_exception.h"
#include "xap_App.h"
#include "xap_Prefs.h"
#include "xap_ModuleManager.h"
#include "xap_Module.h"
#include "UserPluginLoader.h"


// the loader manages instances of one of these target classes
// pasted from xap_ModuleManager.cpp but this is only a demo plugin ...

#if defined (WIN32)
  #include "xap_Win32Module.h"
  #define MODULE_CLASS XAP_Win32Module

#elif defined (__BEOS__)
  #include "xap_BeOSModule.h"
  #define MODULE_CLASS XAP_BeOSModule

#elif defined (__QNXNTO__)
  #include "xap_QNXModule.h"
  #define MODULE_CLASS XAP_QNXModule

#elif defined (__APPLE__) && defined (XP_MAC_TARGET_MACOSX)
	#if defined (XP_MAC_TARGET_CARBON)
		#include <ConditionalMacros.h>
		#if defined(TARGET_RT_MAC_CFM) && (TARGET_RT_MAC_CFM == 1)
			#include "xap_MacCFMModule.h"
			#define MODULE_CLASS XAP_MacModule
		#elif defined (TARGET_RT_MAC_MACHO) && (TARGET_RT_MAC_MACHO == 1)
			#include "xap_MacModule.h"
			#define MODULE_CLASS XAP_MacModule
		#else
			#error Unknown Apple architecture
		#endif
	#elif defined (XP_TARGET_COCOA)
		#include "xap_CocoaModule.h"
		#define MODULE_CLASS XAP_CocoaModule	
	#endif
#else
  // *NIX
  #include "xap_UnixModule.h"
  #define MODULE_CLASS XAP_UnixModule

#endif


// todo
#undef UT_DEBUGMSG
#define UT_DEBUGMSG(M) printf M


// log information about plugin loading into the <log> section of AbiWord.profile
// (we save the prefs file after each call, so as to maximise the information we have in
// case a plugin crashes and the crash handler does not get a chance to save it for us
#define USERPLUGINLOADER_LOG(msg1, msg2)                                               \
if (XAP_App::getApp() && XAP_App::getApp()->getPrefs ()) {                             \
    UT_String __s;                                                                     \
    UT_String_sprintf (__s, "(L%d): %s %s", __LINE__, msg1, msg2);                     \
	UT_DEBUGMSG(("ROB: UserPluginLoader '%s'\n",__s.c_str ()));                        \
    XAP_App::getApp()->getPrefs()->log ("UserPluginLoader::loadModule", __s.c_str ()); \
	XAP_App::getApp()->getPrefs()->savePrefsFile ();                                   \
}

/*!
* Pluggable module loader implementation.
*/
XAP_Module * 
UserPluginLoader::loadModule (const char * szFilename) 
{	
	// FIXME this is not really xp but since this is a demo plugin ...
	if (UT_strnicmp (szFilename, "/home/", 6) != 0) {
		UT_DEBUGMSG (("ROB: UserPluginLoader: not responsible\n"));
		return NULL;
	}

	UT_DEBUGMSG (("ROB: UserPluginLoader: '%s'\n", szFilename));
	XAP_Module * pModule = NULL;

	UT_TRY {
		pModule = new MODULE_CLASS;
	}
	UT_CATCH (...) {
		UT_DEBUGMSG (("ROB: UserPluginLoader: could not instantiate\n"));
		pModule = NULL;
		return NULL;
	}

	if (!pModule->load (szFilename)) {

		USERPLUGINLOADER_LOG ("Failed to load module", szFilename);
		
		char * errorMsg = 0;
		if (pModule->getErrorMsg (&errorMsg)) {
			USERPLUGINLOADER_LOG ("Error msg", errorMsg);
			FREEP (errorMsg);
		}
		delete pModule;
		pModule = NULL;
		return NULL;
	}

	UT_DEBUGMSG (("ROB: UserPluginLoader: loaded pModule='%d'\n", pModule));	
	return pModule;
}


/*!
* Instance of loaded plugin.
*/
static UserPluginLoader *UserPluginLoader_instance = NULL;


/*!
* Initialisation of the plugin.
*/
bool
UserPluginLoader_registerFactory () 
{	
	bool bRegistered = FALSE;

	if (UserPluginLoader_instance == NULL) {
		UserPluginLoader *pUPL = new UserPluginLoader ();
		bRegistered = XAP_ModuleManager::instance().registerFactory (pUPL);
		if (bRegistered) {
			UserPluginLoader_instance = pUPL;
		}
	}

	UT_DEBUGMSG (("ROB: UserPluginLoader_registerFactory () '%d'\n", bRegistered));
	return bRegistered;
}

/*!
* Cleaning up the plugin.
*/
bool
UserPluginLoader_unregisterFactory () 
{
	bool bUnregistered = FALSE;

	if (UserPluginLoader_instance != NULL) {
		bUnregistered = XAP_ModuleManager::instance().unregisterFactory (UserPluginLoader_instance);
		if (bUnregistered) {
			FREEP (UserPluginLoader_instance);
			UserPluginLoader_instance = NULL;
		}
	}

	UT_DEBUGMSG (("ROB: UserPluginLoader_unregisterFactory () '%d'\n", bUnregistered));
	return bUnregistered;
}



ABI_PLUGIN_DECLARE	("UserPluginLoader")

/*!
* Standard abi plugin interface.
*/
ABI_FAR_CALL
int abi_plugin_register (XAP_ModuleInfo * mi)
{
    mi->name = "User Plugin Loader";
    mi->desc = "Loader for user-specific AbiWord plugins.";
    mi->version = ABI_VERSION_STRING;
    mi->author = "Robert Staudinger";
    mi->usage = "No Usage";
    
	UserPluginLoader_instance = NULL;
    return UserPluginLoader_registerFactory ();
}

/*!
* Standard abi plugin interface.
*/
ABI_FAR_CALL
int abi_plugin_unregister (XAP_ModuleInfo * mi)
{
    mi->name = 0;
    mi->desc = 0;
    mi->version = 0;
    mi->author = 0;
    mi->usage = 0;

    return UserPluginLoader_unregisterFactory ();
}

/*!
* Standard abi plugin interface.
*/
ABI_FAR_CALL
int abi_plugin_supports_version (UT_uint32 major, UT_uint32 minor, UT_uint32 release)
{
    return 1; 
}
