/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/*
 * Copyright (C) 2004 Francis James Franklin
 * Copyright (C) 2001 Dom Lachowicz & Michael D. Pritchett
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef ABI_PLUGIN_BUILTIN
#define abi_plugin_register abipgn_urldict_register
#define abi_plugin_unregister abipgn_urldict_unregister
#define abi_plugin_supports_version abipgn_urldict_supports_version
#endif

#include <ctype.h>

#include "ut_string_class.h"

#include "xap_App.h"
#include "xap_Frame.h"
#include "xap_Menu_Layouts.h"
#include "xap_Module.h"

#include "ev_EditMethod.h"
#include "ev_Menu.h"
#include "ev_Menu_Actions.h"
#include "ev_Menu_Labels.h"
#include "ev_Menu_Layouts.h"

#include "fv_View.h"

#include "ap_Menu_Id.h"

// -----------------------------------------------------------------------
// -----------------------------------------------------------------------

static bool s_append_query (UT_UTF8String & URL, AV_View * v, EV_EditMethodCallData * d)
{
	XAP_Frame * pFrame = XAP_App::getApp()->getLastFocussedFrame();
	if (!pFrame)
		return false;

	FV_View * pView = static_cast<FV_View*>(pFrame->getCurrentView());
	if (!pView)
		return false;

	if (pView->isSelectionEmpty())
		{
			/* If the user is on a word, but does not have it selected, we need
			 * to go ahead and select that word so that the search/replace goes
			 * correctly.
			 */
			pView->moveInsPtTo(FV_DOCPOS_EOW_MOVE);
			pView->moveInsPtTo(FV_DOCPOS_BOW);
			pView->extSelTo(FV_DOCPOS_EOW_SELECT);   
		}
	if (pView->isSelectionEmpty())
		return false;

	UT_UCS4Char * ucs4 = 0;

	pView->getSelectionText(*&ucs4);

	bool bHaveText = false;

	if (ucs4)
		{
			char buf[3];

			buf[0] = '+';
			buf[2] = 0;

			bool bInsertSpace = false;
  
			UT_UCS4Char * ptr = ucs4;

			while (*ptr)
				{
					if ((*ptr & 0x7f) == *ptr)
						{
							int ic = static_cast<int>(*ptr & 0x7f);

							buf[1] = static_cast<char>(ic);

							if (isalnum(ic) || (buf[1] == '-'))
								{
									URL += ((bInsertSpace && bHaveText) ? buf : buf + 1);

									bInsertSpace = false;
									bHaveText = true;
								}
							else
								{
									bInsertSpace = true;
								}
						}
					else
						{
							bInsertSpace = true;
						}
					++ptr;
				}
		}
	return bHaveText;
}

static const char * s_DictOrg_MethodName  = "AbiURLDict_DictOrg";
static const char * s_DictOrg_MenuLabel   = "DICT.org &URL Dictionary";
static const char * s_DictOrg_MenuTooltip = "Queries DICT.org, a dictionary website";

static bool s_DictOrg_invoke (AV_View * v, EV_EditMethodCallData * d)
{
	UT_UTF8String URL("http://www.dict.org/bin/Dict?Form=Dict1&Database=*&Strategy=*&Query=");

	bool okay = false;

	if (s_append_query(URL, v, d))
		{
			okay = XAP_App::getApp()->openURL(URL.utf8_str());
		}
	return okay;
}

static const char * s_Webster_MethodName  = "AbiURLDict_Webster";
static const char * s_Webster_MenuLabel   = "Webster Hypertext Gateway";
static const char * s_Webster_MenuTooltip = "Queries the Webster Hypertext Gateway website";
 
static bool s_Webster_invoke (AV_View * v, EV_EditMethodCallData * d)
{
	UT_UTF8String URL("http://www.bennetyee.org/http_webster.cgi?");

	bool okay = false;

	if (s_append_query(URL, v, d))
		{
			okay = XAP_App::getApp()->openURL(URL.utf8_str());
		}
	return okay;
}

static const char * s_Cambridge_MethodName  = "AbiURLDict_Cambridge";
static const char * s_Cambridge_MenuLabel   = "Cambridge On-line Dictionary";
static const char * s_Cambridge_MenuTooltip = "Queries the Cambridge dictionary website";
 
static bool s_Cambridge_invoke (AV_View * v, EV_EditMethodCallData * d)
{
	UT_UTF8String URL("http://dictionary.cambridge.org/results.asp?searchword=");

	bool okay = false;

	if (s_append_query(URL, v, d))
		{
			okay = XAP_App::getApp()->openURL(URL.utf8_str());
		}
	return okay;
}

static const char * s_Oxford_MethodName  = "AbiURLDict_Oxford";
static const char * s_Oxford_MenuLabel   = "Oxford On-line English Dictionary";
static const char * s_Oxford_MenuTooltip = "Queries the Oxford English Dictionary website";
 
static bool s_Oxford_invoke (AV_View * v, EV_EditMethodCallData * d)
{
	UT_UTF8String URL("http://dictionary.oed.com/cgi/findword?query_type=word&queryword=");

	bool okay = false;

	if (s_append_query(URL, v, d))
		{
			okay = XAP_App::getApp()->openURL(URL.utf8_str());
		}
	return okay;
}

static void s_removeFromMenus (const char * method_name, const char * menu_label)
{
	XAP_App * pApp = XAP_App::getApp();

	/* remove the edit method
	 */
	EV_EditMethodContainer * pEMC = pApp->getEditMethodContainer();
	EV_EditMethod * pEM = ev_EditMethod_lookup(method_name);
	if (!pEM)
		{
			// huh. presumably it hasn't been added...
			return;
		}
	pEMC->removeEditMethod(pEM);
	DELETEP(pEM);

	/* now remove from the menus
	 */
	XAP_Menu_Factory * pFact = pApp->getMenuFactory();

	pFact->removeMenuItem("Main",        NULL, menu_label);
	pFact->removeMenuItem("contextText", NULL, menu_label);

	int frameCount = pApp->getFrameCount();

	for (int i = 0; i < frameCount; ++i)
		{
			/* Get the current frame that we're iterating through.
			 */
			XAP_Frame * pFrame = pApp->getFrame(i);

			pFrame->rebuildMenus();
		}
}

static void s_removeAllFromMenus ()
{
	s_removeFromMenus(s_Oxford_MethodName,    s_Oxford_MenuLabel);
	s_removeFromMenus(s_Cambridge_MethodName, s_Cambridge_MenuLabel);
	s_removeFromMenus(s_Webster_MethodName,   s_Webster_MenuLabel);
	s_removeFromMenus(s_DictOrg_MethodName,   s_DictOrg_MenuLabel);
}

static void s_addToMenus(const char * method_name, EV_EditMethod_pFn method_func, const char * menu_label, const char * menu_tooltip)
{
	XAP_App * pApp = XAP_App::getApp();
    
	/* Create an EditMethod that will link our method's name with
	 * it's callback function.  This is used to link the name to 
	 * the callback.
	 */
	EV_EditMethod * myEditMethod = new EV_EditMethod(
													 method_name,  // name of callback function
													 method_func,  // callback function itself.
													 0,            // no additional data required.
													 ""            // description -- allegedly never used for anything
													 );

	/* Now we need to get the EditMethod container for the application.
	 * This holds a series of Edit Methods and links names to callbacks.
	 */
	EV_EditMethodContainer * pEMC = pApp->getEditMethodContainer();
  
	/* We have to add our EditMethod to the application's EditMethodList
	 * so that the application will know what callback to call when a call
	 * to [method_func] is received.
	 */
	pEMC->addEditMethod(myEditMethod);

	/* Now we need to grab an ActionSet.  This is going to be used later
	 * on in our for loop.  Take a look near the bottom.
	 */
	EV_Menu_ActionSet * pActionSet = pApp->getMenuActionSet();

	/* We need to go through and add the menu element to each "frame" 
	 * of the application.  We can iterate through the frames by doing
	 * XAP_App::getFrameCount() to tell us how many frames there are,
	 * then calling XAP_App::getFrame(i) to get the i-th frame.
	 */

	XAP_Menu_Factory * pFact = pApp->getMenuFactory();

	/* Put it in the context menu.
	 */
	XAP_Menu_Id newID = pFact->addNewMenuAfter("contextText", NULL, "Bullets and &Numbering", EV_MLF_Normal);

	pFact->addNewLabel(NULL, newID, menu_label, menu_tooltip);

	/* Also put it under word Wount in the main menu,
	 */
	pFact->addNewMenuAfter("Main", NULL, "&Word Count", EV_MLF_Normal, newID);

	/* Create the Action that will be called.
	 */
	EV_Menu_Action * myAction = new EV_Menu_Action(
												   newID,        // id that the layout said we could use
												   0,            // no, we don't have a sub menu.
												   0,            // no, we don't raise a dialog.
												   0,            // no, we don't have a checkbox.
												   0,
												   method_name,  // name of callback function to call.
												   NULL,         // don't know/care what this is for
												   NULL          // don't know/care what this is for
												   );

	/* Now what we need to do is add this particular action to the ActionSet
	 * of the application.  This forms the link between our new ID that we 
	 * got for this particular frame with the EditMethod that knows how to 
	 * call our callback function.
	 */
	pActionSet->addAction(myAction);

	int frameCount = pApp->getFrameCount();

	for (int i = 0; i < frameCount; ++i)
		{
			/* Get the current frame that we're iterating through.
			 */
			XAP_Frame * pFrame = pApp->getFrame(i);

			pFrame->rebuildMenus();
		}
}

static void s_addAllToMenus ()
{
	s_addToMenus(s_Oxford_MethodName,    s_Oxford_invoke,    s_Oxford_MenuLabel,    s_Oxford_MenuTooltip);
	s_addToMenus(s_Cambridge_MethodName, s_Cambridge_invoke, s_Cambridge_MenuLabel, s_Cambridge_MenuTooltip);
	s_addToMenus(s_Webster_MethodName,   s_Webster_invoke,   s_Webster_MenuLabel,   s_Webster_MenuTooltip);
	s_addToMenus(s_DictOrg_MethodName,   s_DictOrg_invoke,   s_DictOrg_MenuLabel,   s_DictOrg_MenuTooltip);
}

// -----------------------------------------------------------------------
//
//      Abiword Plugin Interface 
//
// -----------------------------------------------------------------------

ABI_FAR_CALL
int abi_plugin_register (XAP_ModuleInfo * mi)
{
	mi->name    = "AbURLDict Web Dictionaries Plug-in";
	mi->desc    = "Web-dictionary support for AbiWord: DICT.org, Webster's Hypertext Gateway, Cambridge dictionaries and Oxford English Dictionary.";
	mi->version = ABI_VERSION_STRING;
	mi->author  = "Michael D. Pritchett, Dom Lachowicz, Francis James Franklin";
	mi->usage   = "Looks up selected word (or words) - valid character range is restricted to ASCII.";

	/* Add the dictionary to AbiWord's menus.
	 */
	s_addAllToMenus();

	return 1;
}

ABI_FAR_CALL
int abi_plugin_unregister (XAP_ModuleInfo * mi)
{
	mi->name    = 0;
	mi->desc    = 0;
	mi->version = 0;
	mi->author  = 0;
	mi->usage   = 0;

	s_removeAllFromMenus();

	return 1;
}

ABI_FAR_CALL
int abi_plugin_supports_version (UT_uint32 major, UT_uint32 minor, UT_uint32 release)
{
	return 1; 
}
