// UT_String.cpp

// A simple string class for use where templates are not
// allowed.
//
// Copyright (C) 2001 Mike Nordell <tamlin@algonet.se>
// 
// This class is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This class is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
// 02111-1307, USA.
//
#include <stdlib.h>
#include <string.h>
#include "UT_String.h"

//
// This string class is carefully crafted to meet the following requirements.
// Don't make changes that violates these carefully selected criterias.
//
// - It shall not cause or use C++ exceptions.
// - It shall not use templates.
// - It shall not provide a sorting order.
// - It shall allow somewhat inferior compilers to use it (this is somewhat
//   arbitrary).
// - It shall work with non-conforming library implementations.
// - It shall not use realloc.
// - It shall not use reference counting since that is 1) not
//   platform independent (the need for some kind of locking mechanism)
//   and 2) in a multi threaded environment every single string would
//   still have to be copied, where the locking just defeats its own
//   purpose, and finally 3) locking slow us down.
//
// These criterias rule out the usage of new/delete since they
// can cause exeptions to be thrown using. They also rule out
// the std::nothrow mechanism for bad libraries. This leaves malloc
// for memory management.
//


//////////////////////////////////////////////////////////////////

static const char pszEmpty[] = "";

// currently just a skeleton allocator to get the code in place

class UT_String_allocator
{
public:
	typedef void* (*pfnAlloc)(size_t n);
	typedef void  (*pfnFree)(void* p);

	UT_String_allocator(pfnAlloc	pAlloc	= &malloc,
						pfnFree		pFree	= &free)
	:	m_pAlloc(pAlloc),
		m_pFree(pFree)
	{
	}

	char*	Alloc_char(size_t n)	const { return (char*)Alloc(n); }
	void*	Alloc(size_t n)			const { return (*m_pAlloc)(n); }
	void	Free(void* p)			const { (*m_pFree)(p); }

private:
	pfnAlloc	m_pAlloc;
	pfnFree		m_pFree;
};

static UT_String_allocator default_allocator;


//////////////////////////////////////////////////////////////////

class UT_StringImpl
{
	// just a glorified data-carrier that is really like a struct
public:
	UT_StringImpl()
	:	psz(pszEmpty),
		len(0),
		allocator(default_allocator)
	{
	}
	UT_StringImpl(const UT_StringImpl& rhs);
	UT_StringImpl(const char* sz);
	~UT_StringImpl()
	{
		if (psz != pszEmpty) {
			Free((char*)psz);	// really const_cast
		}
	}

	UT_Bool			dup(const UT_StringImpl& rhs);
	UT_Bool			dup(const char* sz);
	void			append(const char* sz);
	size_t			size() const { return len; }
	const char*		data() const { return psz; }

private:
	char*	Alloc(size_t n) { return allocator.Alloc_char(n); }
	void	Free(void* p) { allocator.Free(p); }
	void	replace(char* p, size_t n);

	const char*				psz;
	size_t					len;
	UT_String_allocator&	allocator;
};

UT_StringImpl::UT_StringImpl(const UT_StringImpl& rhs)
:	psz(pszEmpty),
	len(0),
	allocator(default_allocator)
{
	dup(rhs);
}

UT_StringImpl::UT_StringImpl(const char* sz)
:	psz(pszEmpty),
	len(0),
	allocator(default_allocator)
{
	dup(sz);
}

UT_Bool UT_StringImpl::dup(const UT_StringImpl& rhs)
{
	char* p = Alloc(rhs.len + 1);
	if (!p) {
		return false;
	}
	memcpy(p, rhs.psz, rhs.len + 1);
	replace(p, rhs.len);
	return true;
}

UT_Bool UT_StringImpl::dup(const char* sz)
{
	const size_t nLen = sz ? strlen(sz) : 0;
	if (nLen) {
		char* p = Alloc(nLen + 1);
		if (p) {
			memcpy(p, sz, nLen + 1);
			replace(p, nLen);
			return true;
		}
	}
	return false;
}

void UT_StringImpl::append(const char* sz)
{
	if (!sz || *sz == '\0') {
		return;
	}
	if (psz == pszEmpty) {
		dup(sz);
		return;
	}
	const size_t nLen1 = size();
	const size_t nLen2 = strlen(sz);
	char* p = Alloc(nLen1 + nLen2 + 1);
	if (p) {
		memcpy(p, psz, nLen1);
		memcpy(p + nLen1, sz, nLen2 + 1);
		replace(p, nLen1 + nLen2);
	}
}

void UT_StringImpl::replace(char* p, size_t n)
{
	if (psz != pszEmpty) {
		Free((char*)psz);	// really const_cast
	}
	psz = p;
	len = n;
}


//////////////////////////////////////////////////////////////////

UT_String::UT_String()
:	pimpl(new UT_StringImpl)
{
}

UT_String::UT_String(const char* sz)
:	pimpl(new UT_StringImpl(sz))
{
}

UT_String::UT_String(const UT_String& rhs)
:	pimpl(new UT_StringImpl(*rhs.pimpl))
{
}

UT_String::~UT_String()
{
	delete pimpl;
}

UT_String& UT_String::operator=(const UT_String& rhs)
{
	if (this != &rhs) {
		pimpl->dup(*rhs.pimpl);
	}
	return *this;
}

UT_String& UT_String::operator=(const char* rhs)
{
	pimpl->dup(rhs);
	return *this;
}

UT_String& UT_String::operator+=(const UT_String& rhs)
{
	pimpl->append(rhs.pimpl->data());
	return *this;
}

UT_String& UT_String::operator+=(const char* rhs)
{
	pimpl->append(rhs);
	return *this;
}


//////////////////////////////////////////////////////////////////
// accessors
size_t UT_String::size() const
{
	return pimpl->size();
}

UT_Bool UT_String::empty() const
{
	return pimpl->size() == 0;
}

UT_Bool UT_String::operator==(const UT_String& rhs) const
{
	return *this == rhs.pimpl->data();
}

UT_Bool UT_String::operator==(const char* rhs) const
{
	return strcmp(pimpl->data(), rhs) == 0;
}

const char* UT_String::c_str() const
{
	return pimpl->data();
}


//////////////////////////////////////////////////////////////////
// End of class members, start of free functions
//////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////
// Helpers

UT_Bool operator==(const char*  s1, const UT_String& s2)
{
	return s2 == s1;
}

UT_Bool operator!=(const UT_String& s1, const UT_String& s2)
{
	return !(s1 == s2);
}

UT_Bool operator!=(const UT_String& s1, const char*  s2)
{
	return !(s1 == s2);
}

UT_Bool operator!=(const char*  s1, const UT_String& s2)
{
	return !(s2 == s1);
}

UT_String operator+(const UT_String& L, const UT_String& R)
{
	UT_String s1(L);
	s1 += R;
	return s1;
}

