/*
 * Copyright 2008 Novamente LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * @author miross
 * Created on Mar 29, 2005
 */
package relex;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.UUID;

import relex.output.SimpleView;

/**
 * This class encapsulates information about an input sentence
 * which is generated by Relex.  
 */
public class RelexInfo implements Serializable
{
	private static final long serialVersionUID = -3047552550351161106L;
	
	private String originalSentence;
	private ArrayList<ParsedSentence> parses;
	private int numParses;

	private String sentenceID;
	
	public RelexInfo()
	{
		originalSentence = null;
		parses = new ArrayList<ParsedSentence>(); // empty list
		numParses = 0;
	}

	public RelexInfo(String os, ArrayList<ParsedSentence> pl)
	{
		originalSentence = os;
		parses = pl;
		assign_id();
		numParses = pl.size();
	}

	/**
	 * Assign a unique sentence ID to each sentence; this is required
	 * for OpenCog output, where each sentence and parse needs to be 
	 * tagged.
	 */
	private void assign_id()
	{
		UUID guid = UUID.randomUUID();
		sentenceID = "sentence_" + guid;

		int n = 0;
		for (ParsedSentence parse: parses)
		{
			String id = sentenceID + "_parse_" + n;
			parse.setIDString(id);
			parse.setRI(this);
			n++;
		}
	}

	public String getID()
	{
		return sentenceID;
	}

	public void setSentence(String s)
	{
		originalSentence = s;
	}
	
	public String getSentence()
	{
		return originalSentence;
	}
	
	public void setParses (ArrayList<ParsedSentence> pl)
	{
		parses = pl;
		assign_id();
	}

	public ArrayList<ParsedSentence> getParses()
	{
		return parses;
	}

	public void setNumParses(int np)
	{
		numParses = np;
	}

	public int getNumParses()
	{
		return numParses;
	}

	public String toString()
	{
		StringBuilder sb = new StringBuilder();
		int pn = 0;
		for (ParsedSentence parse: parses)
		{
			pn++;
			sb.append("Parse " + pn + " of " + parses.size() + "\n");
			sb.append(SimpleView.printRelations(parse)).append("\n");
		}
		return sb.toString();
	}
}

