/* Copyright (C) 2007 One Laptop Per Child
 * Author: Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <dirent.h> // FIXME: not portable
#include <string>

#include "xap_App.h"
#include "ut_assert.h"
#include "ut_debugmsg.h"

#include "LanguageRepository.h"
#include "LanguageDefinition.h"

static int s_langfilter(const struct dirent* f)
{
	UT_return_val_if_fail(f->d_name, 0);
	std::string name(f->d_name);
	std::string::size_type pos = name.find(".lang", name.size()-5);
	if (pos != std::string::npos)
		return 1;
	return 0;
}

// TODO: for now, we load all languages completely; try to do this more lazy
LanguageRepository::LanguageRepository()
{
	// look for global language files
	gchar* path = g_build_filename(XAP_App::getApp()->getAbiSuiteLibDir(), "language-specs", NULL);
	if (path)
	{
		_loadAll(path);
		g_free(path);
	}

	// look for user specific language files
	path = g_build_filename(XAP_App::getApp()->getUserPrivateDirectory(), "language-specs", NULL);
	if (path)
	{
		_loadAll(path);
		g_free(path);
	}
}

LanguageRepository::~LanguageRepository()
{
}

void LanguageRepository::_loadAll(const string& path)
{
	xxx_UT_DEBUGMSG(("LanguageRepository::_loadAll() - path: %s\n", path.c_str()));

	struct dirent **namelist;

	// FIXME: use a portable way of finding files

	UT_sint32 n = scandir(path.c_str(), &namelist, s_langfilter, alphasort);
	if (n < 0)
		UT_DEBUGMSG(("Scandir error for path: %s\n", path.c_str()));
	else
	{
		while (n--)
		{
			gchar* lang_path = g_build_filename(path.c_str(), namelist[n]->d_name, (void*)0);
			if (lang_path)
			{
				LanguageDefinition* pLD = LanguageDefinition::construct(lang_path);
				if (pLD)
				{
					map<string, LanguageDefinition*>::iterator pos = m_vLangs.find(pLD->getLanguage());
					if (pos == m_vLangs.end())
					{
						xxx_UT_DEBUGMSG(("Adding language definition: %s\n", pLD->getLanguage().c_str()));
						m_vLangs.insert(map<string, LanguageDefinition*>::value_type(pLD->getLanguage(), pLD));
					}
					else
					{
						xxx_UT_DEBUGMSG(("Skipping duplicate language definition: %s\n", pLD->getLanguage().c_str()));
						DELETEP(pLD);
					}
				}
				g_free(lang_path);
			}
			free(namelist[n]);
		}
		free(namelist);
	}
}

const LanguageDefinition* LanguageRepository::getLanguageDefinitionByName(const string& name)
{
	map<string, LanguageDefinition*>::iterator pos = m_vLangs.find(name);
	if (pos != m_vLangs.end())
	{
		return (*pos).second;
	}
	return 0;
}
