/*
 * AbiCollab - Code to enable the modification of remote documents.
 * Copyright (C) 2005 by Martin Sevior
 * Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <string>
#include <vector>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xmlreader.h>

#include "ut_vector.h"
#include "pd_Document.h"
#include "px_ChangeRecord.h"
#include "px_CR_SpanChange.h"
#include "px_CR_FmtMarkChange.h"  
#include "px_CR_SpanChange.h"
#include "px_CR_FmtMark.h"        
#include "px_CR_Span.h"
#include "px_CR_Glob.h"           
#include "px_CR_StruxChange.h"
#include "px_CR_ObjectChange.h"   
#include "px_CR_Strux.h"
#include "px_CR_Object.h"
#include "AbiCollab_Packet.h"
#include "pt_Types.h"

#include <handlers/xp/Buddy.h>

static const gchar * szAbiCollab_Packet_PTName[] =
{
    PT_STYLE_ATTRIBUTE_NAME		,
    PT_LEVEL_ATTRIBUTE_NAME		,
    PT_LISTID_ATTRIBUTE_NAME	,
    PT_PARENTID_ATTRIBUTE_NAME	,
    PT_NAME_ATTRIBUTE_NAME		,
    PT_TYPE_ATTRIBUTE_NAME		,
    PT_BASEDON_ATTRIBUTE_NAME	,
    PT_FOLLOWEDBY_ATTRIBUTE_NAME,
    PT_ID_ATTRIBUTE_NAME           ,
    PT_HEADER_ATTRIBUTE_NAME	   ,
    PT_HEADEREVEN_ATTRIBUTE_NAME,
    PT_HEADERFIRST_ATTRIBUTE_NAME,
    PT_HEADERLAST_ATTRIBUTE_NAME,
    PT_FOOTER_ATTRIBUTE_NAME	   ,
    PT_FOOTEREVEN_ATTRIBUTE_NAME,
    PT_FOOTERFIRST_ATTRIBUTE_NAME,
    PT_FOOTERLAST_ATTRIBUTE_NAME,
    PT_REVISION_ATTRIBUTE_NAME     ,
    PT_ID_ATTRIBUTE_NAME           ,
    PT_STRUX_IMAGE_DATAID          ,
    PT_XID_ATTRIBUTE_NAME          ,
    PT_DATAITEM_ATTRIBUTE_NAME          ,
    PT_IMAGE_DATAID,
    PT_IMAGE_TITLE ,
    PT_IMAGE_DESCRIPTION
};

SessionPacketFactory::SessionPacketFactory(const RawPacket& data)
	: m_pDocReader(NULL),
	m_pCurNode(NULL)
{
	UT_DEBUGMSG(("SessionPacketFactory() - packet data is |%s| \n", data.packet.utf8_str()));
	m_pDocReader = xmlReadMemory(data.packet.utf8_str(), strlen(data.packet.utf8_str()), 0, "UTF-8", 0);
	m_pCurNode = xmlDocGetRootElement(m_pDocReader);
}

SessionPacketFactory::SessionPacketFactory(xmlNode* sessionNode)
	: m_pDocReader(NULL),
	m_pCurNode(sessionNode)
{
}

SessionPacketFactory::~SessionPacketFactory(void)
{
	if (m_pDocReader)
		xmlFreeDoc(m_pDocReader);
}

SessionPacket* SessionPacketFactory::_getNextPacket(xmlNode* pPacketNode, const UT_UTF8String& sSessionId)
{
	UT_return_val_if_fail(pPacketNode, NULL);
	
	if (pPacketNode->type == XML_ELEMENT_NODE)
	{
		if(strcmp(reinterpret_cast<const char*>(pPacketNode->name), "sig") == 0)
		{
			char* sz = (char *)xmlGetProp(pPacketNode,(const xmlChar *) "type");
			SessionPacket* pPacket = new SignalSessionPacket(sSessionId, getSignalType(sz));
			FREEP(sz);
			return pPacket;
		}
		else if((strcmp(reinterpret_cast<const char*>(pPacketNode->name), "cr") == 0) || (strcmp(reinterpret_cast<const char*>(pPacketNode->name), "pop") == 0))
		{
			UT_UTF8String sType;
			PX_ChangeRecord::PXType cType =  PX_ChangeRecord::PXT_UpdateLayout;
			UT_UTF8String sDocUUID;
			UT_UTF8String sMyUUID;
			UT_UTF8String sCRNum;
			UT_UTF8String sImpCRNum;
			UT_UTF8String sPos;
			UT_UTF8String sGLOB;
			UT_UTF8String sProps;

			UT_sint32 iCRNum = 0;
			PT_DocPosition iPos = 0;
			UT_sint32 iImpCRNum = 0;
			UT_Byte iGLOB = 0;

			size_t iLength= 0;
			char* sz = NULL;
			
			sz = (char *)xmlGetProp(pPacketNode,(const xmlChar *) "type");
			sType = sz;
			xxx_UT_DEBUGMSG(("For Type |%s| \n", sz));
			FREEP(sz);
	
			sz = (char *)xmlGetProp(pPacketNode,(const xmlChar *) "docUUID");
			sDocUUID = sz;
			xxx_UT_DEBUGMSG(("For docUUID |%s| \n", sz));
			FREEP(sz);

			sz = (char *)xmlGetProp(pPacketNode,(const xmlChar *) "myUUID");
			sMyUUID = sz;
			xxx_UT_DEBUGMSG(("For myUUID |%s| \n", sz));
			FREEP(sz);

			sz = (char *)xmlGetProp(pPacketNode,(const xmlChar *) "CRNum");
			sCRNum = sz;
			xxx_UT_DEBUGMSG(("For CRNum |%s| \n", sz));
			FREEP(sz);
			if(sCRNum.size() > 0)
				iCRNum = atoi(sCRNum.utf8_str());

			sz = (char *)xmlGetProp(pPacketNode,(const xmlChar *) "IMPCRNum");
			sImpCRNum = sz;
			xxx_UT_DEBUGMSG(("For IMPCRNum |%s| \n", sz));
			FREEP(sz);
			if(sImpCRNum.size() > 0)
				iImpCRNum = atoi(sImpCRNum.utf8_str());
	
			sz = (char *)xmlGetProp(pPacketNode,(const xmlChar *) "globtype");
			sGLOB = sz;
			xxx_UT_DEBUGMSG(("For GLOB |%s| \n", sz));
			FREEP(sz);
			if(sGLOB.size() > 0)
				iGLOB = atoi(sGLOB.utf8_str());

			sz = (char *)xmlGetProp(pPacketNode,(const xmlChar *) "pos");
			sPos = sz;
			xxx_UT_DEBUGMSG(("For Pos |%s| \n", sz));
			FREEP(sz);
			if(sPos.size() > 0)
				iPos = atoi(sPos.utf8_str());
	    
			bool b = getChangeRecordType(sType, cType);
			if(!b)
			{
				UT_DEBUGMSG(("No changeRecord for |%s| return NULL \n", sType.utf8_str()));
				return NULL;
			}
			xxx_UT_DEBUGMSG(("For CR Type |%s| value %d \n", sType.utf8_str(), cType));
			
			SessionPacket* pPacket = NULL;
			if (strcmp(reinterpret_cast<const char*>(pPacketNode->name), "pop") == 0)
			{
				pPacket = new ChangeRecordSessionPacket(sSessionId, cType, sDocUUID, sMyUUID, iCRNum, iImpCRNum, iPos, 0 /* TODO: is this valid? */, PACKET_CHANGERECORD_POPULATE); 
			}
			else if (strcmp(reinterpret_cast<const char*>(pPacketNode->name), "cr") == 0)
			{
				pPacket = new ChangeRecordSessionPacket(sSessionId, cType, sDocUUID, sMyUUID, iCRNum, iImpCRNum, iPos, iGLOB, PACKET_CHANGERECORD_CR); 
				xxx_UT_DEBUGMSG(("CRType = %d \n", static_cast<ChangeRecordSessionPacket *>(pPacket)->getPXType()));
			}
		
			xmlNode * child = NULL;
			for (child = pPacketNode->children; child; child = child->next)
			{
				if (child->type == XML_ELEMENT_NODE)
				{
					ChangeRecordSessionPacket* pCrsp = static_cast<ChangeRecordSessionPacket*>(pPacket);
					sz = (char *)xmlGetProp(child, (const xmlChar *)"props");
					sProps = sz;
					FREEP(sz);
					pCrsp->fillPropsFromString(sProps);
					pCrsp->fillPTAttributes(child);
	
					pCrsp->setFrag(child->name);
					xxx_UT_DEBUGMSG(("pCrsp %x pos %d \n", pCrsp, pCrsp->getPos()));
					sz = (char *)xmlGetProp(child, (const xmlChar *)"length");
					if (sz)
					{
						UT_UTF8String sLength = sz;
						FREEP(sz);
						iLength = atoi(reinterpret_cast<const char*>(sLength.utf8_str()));
						UT_ASSERT(iLength >= 0);
						if (iLength <= 0)
							iLength = 0; // safety measure
						sz = reinterpret_cast<char*>(xmlNodeGetContent(child));
						pCrsp->setSpanProps(iLength,sz);
						FREEP(sz);
					}
					break;
				}
			}
			return pPacket;
		}
		else if (strcmp(reinterpret_cast<const char*>(pPacketNode->name), "glob") == 0)
		{
			UT_DEBUGMSG(("Got a glob packet\n"));
			GlobSessionPacket* pPacket = new GlobSessionPacket(sSessionId);
			xmlNode* globContentNode = pPacketNode->children;
			while (globContentNode)
			{
				if (globContentNode->type == XML_ELEMENT_NODE)
				{
					SessionPacket* pGlobPacket = _getNextPacket(globContentNode, sSessionId);
					if (pGlobPacket)
						pPacket->addPacket(pGlobPacket);
					else
						UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
				}
				else
					UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
				globContentNode = globContentNode->next;
			}
			return pPacket;
		}
	}
	else
		UT_DEBUGMSG(("Invalid xml session packet: node expected!\n"));

	return NULL;
}

SessionPacket* SessionPacketFactory::_getNextSessionPacket(xmlNode* pSessionNode)
{
	UT_return_val_if_fail(pSessionNode, NULL);

	if (pSessionNode->type == XML_ELEMENT_NODE)
	{
		if (strcmp(reinterpret_cast<const char*>(pSessionNode->name), "session") == 0)
		{
			UT_UTF8String sSessionId = (char*) xmlGetProp(pSessionNode, (const xmlChar *) "id");
			UT_DEBUGMSG(("Got a session packet with id: %s\n", sSessionId.utf8_str()));

			// at the moment, we only allow one ChangeRecord/Signal/Glob per abicollab packet
			xmlNode* pPacketNode = pSessionNode->children;
			return _getNextPacket(pPacketNode, sSessionId);
		}
		else
			UT_DEBUGMSG(("Invalid packet encountered: missing session tag (encountered |%s| instead!\n", pSessionNode->name));
	}
	else
		UT_DEBUGMSG(("Invalid xml session packet: node expected!\n"));

	return NULL;
}


/*!
 * Return the AbiCollab packet from the collection it acquired during
 * construction.
 * Returns NULL when there are no more packets in the collection
 */
SessionPacket* SessionPacketFactory::getNextPacket(void)
{
	if (!m_pCurNode)
		return NULL;
	SessionPacket* pPacket = _getNextSessionPacket(m_pCurNode);
	m_pCurNode = m_pCurNode->next;
	return pPacket;
}

bool SessionPacketFactory::getChangeRecordType(const UT_UTF8String& sType, 
				  PX_ChangeRecord::PXType & crType)
{
	if (sType=="PXT_GlobMarker")
	{
		crType = PX_ChangeRecord::PXT_GlobMarker;
	}
	else if (sType=="PXT_InsertSpan")
	{
		UT_DEBUGMSG(("Found Insertspan CR \n"));
		crType = PX_ChangeRecord::PXT_InsertSpan;
	}
	else if (sType=="PXT_DeleteSpan")
	{
		crType = PX_ChangeRecord::PXT_DeleteSpan;
	}
	else if (sType=="PXT_ChangeSpan")
	{
		crType = PX_ChangeRecord::PXT_ChangeSpan;
	}
	else if (sType=="PXT_InsertStrux")
	{
		crType = PX_ChangeRecord::PXT_InsertStrux;
	}
	else if (sType=="PXT_DeleteStrux")
	{
		crType = PX_ChangeRecord::PXT_DeleteStrux;
	}
	else if (sType=="PXT_ChangeStrux")
	{
		crType = PX_ChangeRecord::PXT_ChangeStrux;
	}
	else if (sType=="PXT_InsertObject")
	{
		crType = PX_ChangeRecord::PXT_InsertObject;
	}
	else if (sType=="PXT_DeleteObject")
	{
		crType = PX_ChangeRecord::PXT_DeleteObject;
	}
	else if(sType=="PXT_ChangeObject")
	{
		crType = PX_ChangeRecord::PXT_ChangeObject;
	}
	else if (sType=="PXT_InsertFmtMark")
	{
		crType = PX_ChangeRecord::PXT_InsertFmtMark;
	}
	else if (sType=="PXT_DeleteFmtMark")
	{
		crType = PX_ChangeRecord::PXT_DeleteFmtMark;
	}
	else if (sType=="PXT_ChangeFmtMark")
	{
		crType = PX_ChangeRecord::PXT_ChangeFmtMark;
	}
	else if (sType=="PXT_ChangePoint")
	{
		crType = PX_ChangeRecord::PXT_ChangePoint;
	}
	else if (sType=="PXT_ListUpdate")
	{
		crType = PX_ChangeRecord::PXT_ListUpdate;
	}
	else if (sType=="PXT_StopList")
	{
		crType = PX_ChangeRecord::PXT_StopList;
	}
	else if(sType=="PXT_UpdateField")
	{
		crType = PX_ChangeRecord::PXT_UpdateField;
	}
	else if (sType=="PXT_RemoveList")
	{
		crType = PX_ChangeRecord::PXT_RemoveList;
	}
	else if (sType=="PXT_UpdateLayout")
	{
		crType = PX_ChangeRecord::PXT_UpdateLayout;
	}
	else if (sType=="PXT_CreateDataItem")
	{
		crType = PX_ChangeRecord::PXT_CreateDataItem;
	}
	else
	{
		return false;
	}
	return true;
}

UT_uint32 SessionPacketFactory::getSignalType(const UT_UTF8String& sType)
{
	UT_uint32 iSignal = 0;
	if (sType == "updatelayout")
	{
		iSignal = PD_SIGNAL_UPDATE_LAYOUT;
	}
	else if (sType == "reformatlayout")
	{
		iSignal = PD_SIGNAL_REFORMAT_LAYOUT;
	}
	else if (sType == "revmodechanged")
	{
		iSignal = PD_SIGNAL_REVISION_MODE_CHANGED;
	}
	else if (sType == "updatepropsrebuild")
	{
		iSignal = PD_SIGNAL_DOCPROPS_CHANGED_REBUILD;
	}
	else if (sType == "updatepropsnorebuild")
	{
		iSignal = PD_SIGNAL_DOCPROPS_CHANGED_NO_REBUILD;
	}
	else if (sType == "docnamechanged")
	{
		iSignal = PD_SIGNAL_DOCNAME_CHANGED;
	}
	else if (sType == "docdirtychanged")
	{
		iSignal = PD_SIGNAL_DOCDIRTY_CHANGED;
	}
	else if (sType == "docsaved")
	{
		iSignal = PD_SIGNAL_DOCSAVED;
	}
	else if (sType == "docclosed")
	{
		iSignal = PD_SIGNAL_DOCCLOSED;
	}
	return iSignal;
}


/* ***************************************************** */
/* *                Packet                               */
/* ***************************************************** */

Packet::Packet() 
	: m_szAtts(NULL),
	m_szProps(NULL)
{
}

Packet::~Packet(void)
{
	_freeAtts();
	_freeProps();
}

char * Packet::getAttribute(char * szName) const
{
	UT_sint32 i = 0;
	char * value = NULL;
	while(m_szAtts && (m_szAtts[i] != NULL) && (strcmp(m_szAtts[i],szName) != 0))
	{
		i += 2;
	}
	if(strcmp(m_szAtts[i],szName) == 0)
	{
		value = m_szAtts[i+1];
	}
	return value;
}

bool Packet::fillPTAttributes(xmlNode * node)
{
	char * value = NULL;
	const gchar * szAtt = NULL;
	_freeAtts();
	UT_GenericVector<const char *> vecAtts;
	UT_sint32 i = 0;
	for(i=0; i < (sizeof(szAbiCollab_Packet_PTName) / sizeof(gchar *));i++)
	{
		szAtt = szAbiCollab_Packet_PTName[i];
		value = (char *)xmlGetProp(node,reinterpret_cast<const xmlChar *>(szAtt));
		if(value != 0)
		{
			xxx_UT_DEBUGMSG(("Found Attribute |%s| Value |%s| \n",szAtt,value));
			vecAtts.addItem(g_strdup(szAtt));
			vecAtts.addItem(g_strdup(value));
			FREEP(value);
		}
	}
	UT_uint32 isize = vecAtts.getItemCount();
	if(isize > 1)
	{
		m_szAtts = new gchar *[isize+1];
		UT_uint32 i = 0;
		for(i=0;i< isize;i++)
		{
			m_szAtts[i] = const_cast<gchar *>(reinterpret_cast<const gchar *>(vecAtts.getNthItem(i)));
			xxx_UT_DEBUGMSG(("String %d value |%s| \n",i,m_szAtts[i]));
		}
		m_szAtts[isize] = 0;
	}
	return true;
}

bool Packet::fillPropsFromString(UT_UTF8String& sProps)
{
	if (sProps.length() == 0)
	{
		return true;	
	}
	xxx_UT_DEBUGMSG(("Prop String is |%s| \n",sProps.utf8_str()));
	int pos = 0;
	int newpos = 0;
	std::string s(sProps.utf8_str());
	UT_GenericVector<std::string *> vProps;
	
	newpos = s.find(";", pos);
	while (newpos != std::string::npos)
	{
	        vProps.addItem(new std::string(s.substr(pos, newpos-pos)));
		pos = newpos+1;
		newpos = s.find(";", pos);
	}
	if (pos != s.length()-1)
	{
	  vProps.addItem(new std::string(s.substr(pos, s.length()-pos)));
	}
	
	int i = 0;
	if(m_szProps)
	{
	    _freeProps();
	}
	m_szProps = new gchar * [vProps.size()*2 + 1];
	UT_uint32 j= 0;
	for (j=0; j < vProps.getItemCount(); j++)
	{
		std::string keyValue = *vProps.getNthItem(j);
		int ppos = keyValue.find(":", 0);
		if (ppos == std::string::npos)
		{
			UT_DEBUGMSG(("Invalid property found!!!\n"));
			_freeProps();
			return false;
		}
		
		m_szProps[i] = g_strdup(keyValue.substr(0, ppos).c_str());
		m_szProps[i+1] = g_strdup(keyValue.substr(ppos+1, keyValue.length()-(ppos+1)).c_str());
		xxx_UT_DEBUGMSG(("Found Prop |%s| Value |%s| \n",m_szProps[i],m_szProps[i+1]));
		i+=2;
	}
	UT_VECTOR_PURGEALL(std::string *,vProps);
	m_szProps[i] = NULL;
	return true;
}

void Packet::_freeProps()
{
	if (m_szProps == NULL)
		return;
	
	UT_sint32 i = 0;
	while(m_szProps[i] != NULL)
	{
		FREEP(m_szProps[i]);
		i++;
	}
	delete [] m_szProps;
	m_szProps = NULL;
}


void Packet::_freeAtts()
{
	if (m_szAtts == NULL)
		return;
	
	UT_sint32 i = 0;
	while(m_szAtts[i] != NULL)
	{
	  xxx_UT_DEBUGMSG(("_freeAtts %d String |%s| \n",i,szAtts[i]));
		FREEP(m_szAtts[i]);
		i++;
	}
	delete [] m_szAtts;
	m_szAtts = NULL;
}

/* ***************************************************** */
/* *                   SessionPacket                     */
/* ***************************************************** */

SessionPacket::SessionPacket(const UT_UTF8String& sSessionId) 
	: Packet(),
	m_sSessionId(sSessionId)
{
}

SessionPacket::~SessionPacket()
{
}

/* ***************************************************** */
/* *          ChangeRecordSessionPacket                  */
/* ***************************************************** */

ChangeRecordSessionPacket::ChangeRecordSessionPacket(
			const UT_UTF8String& sSessionId,
			PX_ChangeRecord::PXType cType, 
			const UT_UTF8String& sDocUUID, 
			const UT_UTF8String& sMyUUID, 
			int iCRNum,
			int iImpCRNum,
			int iPos,
			UT_Byte iGLOB,
			PacketSessionType crType)
	: SessionPacket(sSessionId),
	  m_cType(cType),
	  m_sDocUUID(sDocUUID),
	  m_sMyUUID(sMyUUID),
	  m_iCRNum(iCRNum),
	  m_iPos(iPos),
	  m_iImpCRNum(iImpCRNum),
	  m_sFrag(""),
	  m_iLength(0),
	  m_sValue(""),
	  m_iGLOBType(iGLOB),
	  m_crType(crType)
{
	UT_ASSERT(m_crType == PACKET_CHANGERECORD_POPULATE || m_crType == PACKET_CHANGERECORD_CR);
}

ChangeRecordSessionPacket::~ChangeRecordSessionPacket(void)
{
}

void ChangeRecordSessionPacket::setSpanProps(size_t iLength, const char* value)
{
	m_iLength = iLength;
	m_sValue = value;
}

/* ***************************************************** */
/* *             SignalSessionPacket                     */
/* ***************************************************** */

SignalSessionPacket::SignalSessionPacket(const UT_UTF8String& sSessionId, UT_uint32 iSignal)
	: SessionPacket(sSessionId),
	m_iSignal(iSignal)
{
}

const UT_UTF8String* SignalSessionPacket::serialize() const
{
	// TODO: use libxml2 to generate proper XML
	
	UT_UTF8String* psPacket = new UT_UTF8String();
	*psPacket += "<sig type=\"";
	
	switch (m_iSignal)
	{
		case PD_SIGNAL_UPDATE_LAYOUT:
			*psPacket += "updatelayout\"";
			break;
		case PD_SIGNAL_REFORMAT_LAYOUT:
			*psPacket += "reformatlayout\"";
			break;
		case PD_SIGNAL_REVISION_MODE_CHANGED:
			*psPacket += "revmodechanged\"";
			*psPacket += "/>";
			//m_pAbiCollab->push(*psPacket); // TODO: EEEEEK! FIX THIS
		
			psPacket->clear();
			*psPacket += "<sig type=\"";
			
			// fall through ...
			
		case PD_SIGNAL_DOCPROPS_CHANGED_REBUILD:
			*psPacket += "updatepropsrebuild\"";
			break;
		case PD_SIGNAL_DOCPROPS_CHANGED_NO_REBUILD:
			*psPacket += "updatepropsnorebuild\"";
			break;
		case PD_SIGNAL_DOCNAME_CHANGED:
			*psPacket += "docnamechanged\"";
			break;
		case PD_SIGNAL_DOCDIRTY_CHANGED:
			*psPacket += "docdirtychanged\"";
			break;
		case PD_SIGNAL_DOCSAVED:
			*psPacket += "docsaved\"";
			break;
		case PD_SIGNAL_DOCCLOSED:
			*psPacket += "docclosed\"";
			break;
		default:
			UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
			break;
	}
	*psPacket += "/>";

	return psPacket;
}

/* ***************************************************** */
/* *             GlobSessionPacket                       */
/* ***************************************************** */

GlobSessionPacket::GlobSessionPacket(const UT_UTF8String& sSessionId)
	: SessionPacket(sSessionId)
{
}

GlobSessionPacket::~GlobSessionPacket()
{
	// TODO: delete the packets we hold
}
