/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiCollab- Code to enable the modification of remote documents.
 * Copyright (C) 2005 by Martin Sevior
 * Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <string>
#include <vector>

#include "AbiCollab_Import.h"
#include "AbiCollab.h"
#include "AbiCollab_Export.h"
#include "pd_Document.h"
#include "px_CR_SpanChange.h"
#include "px_CR_FmtMarkChange.h"  
#include "px_CR_SpanChange.h"
#include "px_CR_FmtMark.h"        
#include "px_CR_Span.h"
#include "px_CR_Glob.h"           
#include "px_CR_StruxChange.h"
#include "px_CR_ObjectChange.h"   
#include "px_CR_Strux.h"
#include "px_CR_Object.h"
#include "xap_App.h"
#include "pd_Style.h"
#include "xap_Frame.h"
#include "fv_View.h"
#include "ut_sleep.h"
#include "pt_Types.h"

ABI_Collab_Import::ABI_Collab_Import(AbiCollab* pAbiCollab, PD_Document* doc):
	m_pDoc(doc),
	m_pAbiCollab(pAbiCollab),
	m_iGlobCounter(0)
{
}

ABI_Collab_Import::~ABI_Collab_Import()
{
        UT_sint32 i = static_cast<UT_sint32>( m_vecAdjusts.getItemCount());
	while(i > 0)
	{
	      delete  m_vecAdjusts.getNthItem(i-1);
	      i--;
	}
}

/*!
 * Run back through packets exported from here to account for internet lag.
 * iCRNum is the last CR number received by the remote from this AbiWord.
 * ipos is the position that the CR was applied in the remote document.
 */
UT_sint32 ABI_Collab_Import::adjustImportPointForCR(UT_sint32 iCRNum, UT_sint32 ipos)
{
	ABI_Collab_Export * pExport = m_pAbiCollab->getExport();
	const UT_GenericVector<ChangeAdjust *> * pExpAdjusts = pExport->getAdjusts();
	UT_sint32 diff = 0;
	UT_sint32 iCount = pExpAdjusts->getItemCount();
	UT_sint32 i = iCount-1;
	UT_DEBUGMSG((" Received CRNum %d pos %d \n",iCRNum,ipos));
	//
	// First scan back to find the CR in our export list the remote
	// has seen.
	//
	bool bFound =false;
	while (i >= 0)
	{
	    ChangeAdjust * pChange = pExpAdjusts->getNthItem(i);
	    if(pChange == NULL)
	          break;
	    UT_DEBUGMSG((" Looking at export num %d CR num %d pos %d adjust %d \n",i,pChange->m_iCRNumber,pChange->m_iDocPos,pChange->m_iAdjust));
	    if(pChange->m_iCRNumber <= iCRNum)
	    {
			bFound = true;
			break;
	    }
	    i--;
	}
	if (!bFound)
		return diff;
	i++;
	while (i < iCount)
	{
	    ChangeAdjust * pChange = pExpAdjusts->getNthItem(i);
	    if (pChange == NULL)
			break;
	    UT_DEBUGMSG((" Adjusting export num %d CR num %d pos %d adjust %d diff %d \n",i,pChange->m_iCRNumber,pChange->m_iDocPos,pChange->m_iAdjust,diff));
	    if (pChange->m_iDocPos <= (ipos + diff))
	    {
			diff += pChange->m_iAdjust;
			UT_DEBUGMSG((" Diff now %d \n",diff));
	    }
	    i++;
	}
	return diff;
}
/*!
 * Take a packet contained with a UT_UTF8string, interpret it's
 * contents and apply the implied operations on the document.
 */
bool ABI_Collab_Import::import(const SessionPacket& packet)
{
	UT_GenericVector<AV_View *> vecViews;
	//UT_usleep(200000); // 0.2 seconds useful for debugging

	vecViews.clear();
	m_pDoc->getAllViews( & vecViews);
	UT_sint32 i = 0;
	for(i=0; i< vecViews.getItemCount(); i++)
	{
		vecViews.getNthItem(i)->setActivityMask(false);
	}

	UT_UTF8String sRealDocname;
	sRealDocname = m_pDoc->getOrigDocUUIDString();
	UT_DEBUGMSG(("Import Result type %d \n", packet.getSessionType()));
	switch (packet.getSessionType())
	{
		case PACKET_GLOB:
			{
				const GlobSessionPacket* gp = static_cast<const GlobSessionPacket*>(&packet);
				for (UT_sint32 j = 0; j < gp->getPackets().getItemCount(); j++)
				{
					SessionPacket* pGlobPacket = gp->getPackets().getNthItem(j);
					if (pGlobPacket)
					{
						bool res = import(*pGlobPacket); // yay for recursion :)
						if (!res)
						{
							UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
						}
					}
				}
				goto cleanupTrue;
			}
		
		case PACKET_SIGNAL:
			{
				const SignalSessionPacket* sp = static_cast<const SignalSessionPacket*>(&packet);
				// TODO: FIX ME FIX ME FIXME
				if(0 /*m_pAbiCollab->isServer() */ )
				{
					if( sp->getSignalType() == PD_SIGNAL_DOCSAVED)
					{
						m_pDoc->save();
						goto cleanupTrue;
					}
				}
				if (sp->getSignalType() == PD_SIGNAL_DOCCLOSED)
				{
					m_pAbiCollab->setClose();
					UT_DEBUGMSG(("Doing close from remote signal \n"));
					goto cleanupTrue;
				}
				m_pDoc->signalListeners(sp->getSignalType());
				goto cleanupTrue;
			}
			
		case PACKET_CHANGERECORD_POPULATE:
			/* unhandled at the moment */
			goto cleanupTrue;
		
		case PACKET_CHANGERECORD_CR:
			{
				const ChangeRecordSessionPacket* crp = static_cast<const ChangeRecordSessionPacket*>(&packet);
				UT_sint32 ipos = static_cast<UT_sint32>(crp->getPos());
				UT_sint32 iCRNum = crp->getCRNum();
				m_pDoc->setMyUUID(crp->getDocUUID().utf8_str());
				
				// Adjust the point to account for internet lag
			       
				UT_sint32 idiff = adjustImportPointForCR(crp->getImpCRNum(),ipos);
				// This is the most recently received CRNum 
				// from the remote AbiWord.

				ChangeAdjust * pChange = new ChangeAdjust(ipos,idiff,crp->getCRNum());
				m_vecAdjusts.addItem(pChange);
				UT_DEBUGMSG(("For CR number %d requested point %d adjustment %d \n",iCRNum,ipos,idiff));
				PT_DocPosition pos = static_cast<PT_DocPosition>(ipos+idiff);
				UT_ASSERT(pos <= getEndOfDoc());
				// todo: remove these temp vars
				UT_UTF8String sFrag = crp->getFrag();
				const gchar** szProps = const_cast<const gchar**>(crp->getProps());
				const gchar** szAtts = const_cast<const gchar**>(crp->getAtts());
				UT_UTF8String sValue = crp->getValue();
				
				PT_DocPosition iPos2 = 0;
				switch(crp->getPXType())
				{
					case PX_ChangeRecord::PXT_GlobMarker:
						UT_DEBUGMSG(("Found GLOB marker \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,crp->getGLOBType());
						if (m_iGlobCounter == 0)
						{
							m_iGlobCounter = 1;
						}
						else
						{
							m_pDoc->createAndSendCR(pos, PX_ChangeRecord::PXT_UpdateLayout, false,0);
							m_iGlobCounter = 0;
						}
						break;
					case PX_ChangeRecord::PXT_InsertSpan:
						{
							UT_DEBUGMSG(("Doing InsertSpan |%s| \n", sValue.utf8_str()));
							UT_UCS4String UCSChars = sValue.ucs4_str();
							m_pDoc->insertSpan(pos,UCSChars.ucs4_str(),UCSChars.length());
						}
						break;
					case PX_ChangeRecord::PXT_DeleteSpan:
						{
							iPos2 = pos + crp->getLength();
							PP_AttrProp *p_AttrProp_Before = NULL;
							UT_uint32 icnt = 0;
							m_pDoc->deleteSpan(pos,iPos2,p_AttrProp_Before,icnt,true);
						}
						break;
					case PX_ChangeRecord::PXT_ChangeSpan:
						{
							if (sFrag == "span")
							{
								iPos2 = pos + crp->getLength();
								if((szProps == NULL) && (szAtts == NULL))
								{
									//
									// This happens if we remove all formats
									// we have to handle this seperately
									//
									// Get style of containing block
									//
									PL_StruxDocHandle sdh = NULL;
									m_pDoc->getStruxOfTypeFromPosition(pos,PTX_Block,&sdh);
									if(!sdh)
									{
										UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
										goto cleanupFalse;
									}
									PD_Style * pStyle = m_pDoc->getStyleFromSDH(sdh);
									if(!pStyle)
									{
										UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
										goto cleanupFalse;
									}
									const char * szName =  pStyle->getName();
									const char * atts[3] = {PT_STYLE_ATTRIBUTE_NAME,szName,NULL};
									m_pDoc->changeSpanFmt(PTC_SetExactly, pos, iPos2, atts, szProps);
									break;
								}
								printf(" 0 |%s| 1 |%s| 2 |%s| \n", szProps[0], szProps[0], szProps[2]);
								m_pDoc->changeSpanFmt(PTC_SetExactly, pos, iPos2, szAtts, szProps);
							}
							else
							{
								goto cleanupFalse;
							}
						}
						break;
					case PX_ChangeRecord::PXT_InsertStrux:
						{
							UT_DEBUGMSG(("AbiCollab -- Doing insertStrux pos %d frag |%s|\n",pos,sFrag.utf8_str()));
							PTStruxType pts;
							if (sFrag == "p")
							{
								pts = PTX_Block;
							}
							else if (sFrag == "section")
							{
								pts = PTX_Section;
							}
							else if (sFrag == "footnote")
							{
								pts = PTX_SectionFootnote;
							}
							else if (sFrag == "endfootnote")
							{
								pts = PTX_EndFootnote;
							}
							else if (sFrag == "toc")
							{
								pts = PTX_SectionTOC;
							}
							else if (sFrag == "endtoc")
							{
								pts = PTX_EndTOC;
							}
							else if (sFrag == "endnote")
							{
								pts = PTX_SectionEndnote;
							}
							else if(sFrag == "endendnote")
							{
								pts = PTX_EndEndnote;
							}
							else if (sFrag == "table")
							{
								pts = PTX_SectionTable;
							}
							else if (sFrag == "endtable")
							{
								UT_DEBUGMSG((">>>>>>>>>>>>>>.. got endtable!!!!!!\n"));
								pts = PTX_EndTable;
							}
							else if (sFrag == "cell")
							{
								pts = PTX_SectionCell;
							}
							else if (sFrag == "endcell")
							{
								pts = PTX_EndCell;
							}
							else if (sFrag == "frame")
							{
								pts = PTX_SectionFrame;
							}
							else if (sFrag == "endframe")
							{
								pts = PTX_EndFrame;
							}
							
							if((szProps != NULL) || (szAtts != NULL))
							{
							        m_pDoc->insertStrux(pos, pts, szAtts,szProps);
							}
							else
							{
								m_pDoc->insertStrux(pos, pts);
							}
						}
						break;
					case PX_ChangeRecord::PXT_DeleteStrux:
						{
							PTStruxType pts;
							if(sFrag== "p")
							{
								pts = PTX_Block;
							}
							else if (sFrag == "section")
							{
								pts = PTX_Section;
							}
							else if (sFrag == "footnote")
							{
								pts = PTX_SectionFootnote;
							}
							else if (sFrag == "endfootnote")
							{
								pts = PTX_EndFootnote;
							}
							else if (sFrag == "toc")
							{
								pts = PTX_SectionTOC;
							}
							else if (sFrag == "endtoc")
							{
								pts = PTX_EndTOC;
							}
							else if (sFrag == "endnote")
							{
								pts = PTX_SectionEndnote;
							}
							else if (sFrag == "endendnote")
							{
								pts = PTX_EndEndnote;
							}
							else if (sFrag == "table")
							{
								pts = PTX_SectionTable;
							}
							else if (sFrag == "endtable")
							{
								pts = PTX_EndTable;
							}
							else if (sFrag == "cell")
							{
								pts = PTX_SectionCell;
							}
							else if (sFrag == "endcell")
							{
								pts = PTX_EndCell;
							}
							else if (sFrag == "frame")
							{
								pts = PTX_SectionFrame;
							}
							else if (sFrag == "endframe")
							{
								pts = PTX_EndFrame;
							}
							m_pDoc->deleteStrux(pos,pts,true);
						}
						break;
					case PX_ChangeRecord::PXT_ChangeStrux:
						{
							PTStruxType pts;
							if(sFrag == "p")
							{
								pts = PTX_Block;
							}
							else if(sFrag == "section")
							{
								pts = PTX_Section;
							}
							else if(sFrag == "footnote")
							{
								pts = PTX_SectionFootnote;
							}
							else if(sFrag == "endfootnote")
							{
								pts = PTX_EndFootnote;
							}
							else if( sFrag == "toc")
							{
								pts = PTX_SectionTOC;
							}
							else if( sFrag == "endtoc")
							{
								pts = PTX_EndTOC;
							}
							else if(sFrag == "endnote")
							{
								pts = PTX_SectionEndnote;
							}
							else if(sFrag == "endendnote")
							{
								pts = PTX_EndEndnote;
							}
							else if(sFrag == "table")
							{
								pts = PTX_SectionTable;
							}
							else if(sFrag == "endtable")
							{
								pts = PTX_EndTable;
							}
							else if(sFrag == "cell")
							{
								pts = PTX_SectionCell;
							}
							else if(sFrag == "endcell")
							{
								pts = PTX_EndCell;
							}
							else if(sFrag == "frame")
							{
								pts = PTX_SectionFrame;
							}
							else if(sFrag == "endframe")
							{
								pts = PTX_EndFrame;
							}
							
							if((szProps == NULL) && (szAtts == NULL) )
							{
								UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
								goto cleanupFalse;
							}
							UT_DEBUGMSG(("Executing ChangeStrux pos= %d \n",pos));
							m_pDoc->changeStruxFmt(PTC_SetExactly, pos, pos, szAtts, szProps, pts);
							
							// TODO: this mask is waaaay to generic
							XAP_Frame *pFrame = XAP_App::getApp()->getLastFocussedFrame();
							FV_View* pView = static_cast<FV_View*>(pFrame->getCurrentView());
							pView->notifyListeners(AV_CHG_TYPING | AV_CHG_FMTCHAR | AV_CHG_FMTBLOCK | AV_CHG_PAGECOUNT | AV_CHG_FMTSTYLE );
						}
						break;
					case PX_ChangeRecord::PXT_InsertObject:
						{
							PTObjectType pto;
							if(sFrag == "image")
							{
								pto = PTO_Image;
								UT_sint32 i = 0;
								for(i=0;szProps[i] != NULL;i++)
								{
									UT_DEBUGMSG(("Image: szProps = |%s| \n",szProps[i]));
								}
							}
							else if(sFrag == "field")
							{
								pto = PTO_Field;
							}
							else if(sFrag == "bookmark")
							{
								pto = PTO_Bookmark;
							}
							else if(sFrag == "hyperlink")
							{
								pto = PTO_Hyperlink;
							}
							else if(sFrag == "math")
							{
								pto = PTO_Math;
							}
							else if(sFrag == "embed")
							{
								pto = PTO_Embed;
							}
			
							if((szProps == NULL) && (szAtts == NULL))
							{
								UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
								goto cleanupFalse;
							}
							m_pDoc->insertObject(pos, pto, szAtts, szProps);
						}
						break;
					case PX_ChangeRecord::PXT_DeleteObject:
						{
							iPos2 = pos + 1;
							PP_AttrProp *p_AttrProp_Before = NULL;
							UT_uint32 icnt = 0;
							m_pDoc->deleteSpan(pos, iPos2, p_AttrProp_Before, icnt, true);
						}
						break;
					case PX_ChangeRecord::PXT_ChangeObject:
						{
							if ((szProps == NULL) && (szAtts == NULL))
							{
								UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
								goto cleanupFalse;
							}
							m_pDoc->changeSpanFmt(PTC_SetExactly, pos, pos + 1, szAtts, szProps);
						}
						break;
					case PX_ChangeRecord::PXT_InsertFmtMark:
						{
							if((szProps == NULL) && (szAtts == NULL))
							{
								// nothing to do here, please move along
								
								// NOTE: why does this happen anyway? 
								// This happens when for example when sending over tables:
								// <cr type="PXT_InsertFmtMark" docUUID="51779860-8d6a-11db-900e-a685fd336747" myUUID="66bf2a26-8d6a-11db-900e-a685fd336747" CRNum="5" IMPCRNum="0" pos="4">
								//      <span/>
								// </cr>
								UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
								goto cleanupFalse;
							}
							m_pDoc->changeSpanFmt(PTC_SetExactly, pos, pos, szAtts, szProps);
						}
						break;
					case PX_ChangeRecord::PXT_DeleteFmtMark:
						{
							m_pDoc->deleteFmtMark(pos);
						}
						break; 
					case PX_ChangeRecord::PXT_ChangeFmtMark:
						{
							if ((szProps == NULL) && (szAtts == NULL))
							{
								UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
								goto cleanupFalse;
							}
							m_pDoc->changeSpanFmt(PTC_SetExactly, pos, pos, szAtts, szProps);
						}
						break;
					case PX_ChangeRecord::PXT_ChangePoint:
						UT_DEBUGMSG(("Change Point CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true, 0);
						break; 
					case PX_ChangeRecord::PXT_ListUpdate:
						UT_DEBUGMSG(("ListUpdate CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break; 
					case PX_ChangeRecord::PXT_StopList:
						UT_DEBUGMSG(("StopList CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break; 
					case PX_ChangeRecord::PXT_UpdateField:
						UT_DEBUGMSG(("UpdateFiled CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break;
					case PX_ChangeRecord::PXT_RemoveList:
						UT_DEBUGMSG(("RemoveList CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break;
					case PX_ChangeRecord::PXT_UpdateLayout:
						UT_DEBUGMSG(("UpdateLayout CR \n"));
						m_pDoc->createAndSendCR(pos, crp->getPXType(), true,0);
						break;
					case PX_ChangeRecord::PXT_CreateDataItem:
						{
							UT_DEBUGMSG(("Doing CreateDataItem |%s| \n", sValue.utf8_str()));
							const char * szNameV = g_strdup(crp->getAttribute(PT_DATAITEM_ATTRIBUTE_NAME));
							const void * pToken = NULL;
							void * pHandle = NULL;
							UT_ByteBuf * pBuf= new UT_ByteBuf();
							pBuf->append(reinterpret_cast<const UT_Byte *>(sValue.utf8_str()),sValue.size());

							m_pDoc->createDataItem(szNameV,true,pBuf,pToken,&pHandle);
							delete pBuf;
						}
						break;
					default:
						break;
				}
			}
			goto cleanupTrue;			
		
		default:
			UT_DEBUGMSG(("ABI_Collab_Import::import called with wrong packet type: %d!\n", packet.getType()));
			goto cleanupFalse;			
	}
	
cleanupTrue:
	{
		bool bDone = false;
		for(i=0; i< vecViews.getItemCount(); i++)
		{
			FV_View * pView = static_cast<FV_View *>( vecViews.getNthItem(i));
			if(pView && !bDone && pView->shouldScreenUpdateOnGeneralUpdate())
			{
				m_pDoc->signalListeners(PD_SIGNAL_UPDATE_LAYOUT);
				bDone = true;
			}
			if(pView)
			{
				pView->fixInsertionPointCoords();
				pView->setActivityMask(true);
			}
		}
	}
	m_pDoc->setMyUUID(sRealDocname.utf8_str());
	return true;
	
cleanupFalse:
	for(i=0; i< vecViews.getItemCount(); i++)
	{
		vecViews.getNthItem(i)->setActivityMask(true);
	}
	m_pDoc->setMyUUID(sRealDocname.utf8_str());
	return false;
}

bool ABI_Collab_Import::getStruxType(UT_UTF8String & sInPacket, 
				     PTStruxType & struxType,
				     UT_UTF8String & sAttributes, 
				     UT_UTF8String & sOutpacket)
{
	return true;
}

bool ABI_Collab_Import::getObjectType(UT_UTF8String & sInPacket,
				      PTObjectType & objectType,
				      UT_UTF8String & sAttributes, 
				      UT_UTF8String & sOutpacket)
{
	return true;
}

PT_DocPosition ABI_Collab_Import::getEndOfDoc(void)
{
        PT_DocPosition posEnd;
        m_pDoc->getBounds(true,posEnd);
        return posEnd;
}
