/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiCollab- Code to enable the modification of remote documents.
 * Copyright (C) 2005 by Martin Sevior
 * Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "AbiCollab_Export.h"
#include "pd_Document.h"
#include "px_CR_SpanChange.h"
#include "px_CR_FmtMarkChange.h"  
#include "px_CR_SpanChange.h"
#include "px_CR_FmtMark.h"        
#include "px_CR_Span.h"
#include "px_CR_Glob.h"           
#include "px_CR_StruxChange.h"
#include "px_CR_ObjectChange.h"   
#include "px_CR_Strux.h"
#include "px_CR_Object.h"
#include "ut_bytebuf.h"
#include "ut_base64.h"
#include "AbiCollab.h"
#include <gsf/gsf-utils.h>
#include "ut_timer.h"
#include "AbiCollabSessionManager.h"

ABI_Collab_Export::ABI_Collab_Export(AbiCollab * pAbiCollab, PD_Document* pDoc) : 
	m_pAbiCollab(pAbiCollab),
	m_pDoc(pDoc),
	m_iGlobCounter(0),
	m_chgMaskCached(0),
	m_bCacheChanges(false),
	m_iSpanIndex(-1),
	m_iBlockIndex(-1),
	m_iSectionIndex(-1),
	m_sPacket(""),
	m_iGlobFlag(0),
	m_bDoingGlob(false)
{
}

ABI_Collab_Export::~ABI_Collab_Export()
{
	UT_DEBUGMSG(("AbiCollab Export deleted %x \n",this));
	UT_sint32 i = static_cast<UT_sint32>( m_vecAdjusts.getItemCount());
	while(i > 0)
	{
		delete  m_vecAdjusts.getNthItem(i-1);
		i--;
	}
}

/*!
 * This method takes a changeRecord and returns a ascii string that
 * describes the changeRecord.
 */
bool ABI_Collab_Export::mapChangeRecordToString(const PX_ChangeRecord * pcr, 
						UT_UTF8String & sCRType, 
						UT_UTF8String & sCRArgs, 
						UT_sint32 & iAdjust)
{
	// initialize the changerecord type and arguments
	sCRType = "";
	sCRArgs = "";
	iAdjust = 0;
	UT_sint32 index = static_cast<UT_sint32>(pcr->getIndexAP());
	switch(pcr->getType())
	{
		case PX_ChangeRecord::PXT_GlobMarker:
			sCRType = "PXT_GlobMarker";
			break;
		case PX_ChangeRecord::PXT_InsertSpan:
			sCRType = "PXT_InsertSpan";
			{
				const PX_ChangeRecord_SpanChange * pcrc = static_cast<const PX_ChangeRecord_SpanChange *> (pcr);
				UT_uint32 iLen = pcrc->getLength();
				
				PT_BufIndex bi = pcrc->getBufIndex();
				const UT_UCS4Char* pChars = m_pDoc->getPointer(bi);
				UT_UTF8String sText;
				sText.appendUCS4(pChars,iLen);
				sText = sText.escapeXML();
				iLen= sText.length();
				UT_UTF8String sPos;
				UT_UTF8String_sprintf(sPos,"%d",iLen);
				sCRArgs += "<span ";
				sCRArgs += "length=\"";
				sCRArgs += sPos;
				sCRArgs += "\">";
				sCRArgs += sText;
				sCRArgs += "</span>";
				iAdjust = pcrc->getLength();
			}
			break;
		case PX_ChangeRecord::PXT_DeleteSpan:
			sCRType = "PXT_DeleteSpan";
			{
				const PX_ChangeRecord_SpanChange * pcrc = static_cast<const PX_ChangeRecord_SpanChange *> (pcr);
				UT_uint32 iLen = pcrc->getLength();
				UT_UTF8String sPos;
				UT_UTF8String_sprintf(sPos,"%d",iLen);
				sCRArgs += "span ";
				sCRArgs += "length=\"";
				sCRArgs += sPos;
				sCRArgs += "\"";		
				iAdjust = -iLen;
			}
			break;
		case PX_ChangeRecord::PXT_ChangeSpan:
			sCRType = "PXT_ChangeSpan";
			sCRArgs += "span ";
			m_iSpanIndex = index;
			sCRArgs += " ";
			mapAttributesToString(index, sCRArgs);
			{	
			       const PX_ChangeRecord_SpanChange * pcrc = static_cast<const PX_ChangeRecord_SpanChange *> (pcr);
			       UT_UTF8String sPos;
			       UT_uint32 iLen = pcrc->getLength();
			       UT_UTF8String_sprintf(sPos,"%d",iLen);
			       sCRArgs += " length=\"";
			       sCRArgs += sPos;
			       sCRArgs += "\"";
			}
			break;
		case PX_ChangeRecord::PXT_InsertStrux:
			sCRType = "PXT_InsertStrux";
			iAdjust = 1;
			{
				const PX_ChangeRecord_Strux * pcrx = static_cast<const PX_ChangeRecord_Strux *> (pcr);
				if(pcrx->getStruxType() == PTX_Block)
				{
					sCRArgs += "p";
					if(index != m_iBlockIndex)
					{
						m_iBlockIndex = index;
						sCRArgs += " ";
						mapAttributesToString(index, sCRArgs);
					}
				}
				else if(pcrx->getStruxType() == PTX_Section)
				{
					sCRArgs += "section";
					if(index != m_iSectionIndex)
					{
						m_iSectionIndex = index;
						sCRArgs += " ";
						mapAttributesToString(index, sCRArgs);
					}
				}
				else if(pcrx->getStruxType() == PTX_SectionFootnote)
				{
					sCRArgs += "footnote ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndFootnote)
				{
					sCRArgs += "endfootnote";
				}
				else if(pcrx->getStruxType() == PTX_SectionTOC)
				{
					sCRArgs += "toc ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndTOC)
				{
					sCRArgs += "endtoc";
				}
				else if(pcrx->getStruxType() == PTX_SectionEndnote)
				{
					sCRArgs += "endnote ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndEndnote)
				{
					sCRArgs += "endendnote";
				}
				else if(pcrx->getStruxType() == PTX_SectionTable)
				{
					sCRArgs += "table ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndTable)
				{
					sCRArgs += "endtable";
				}
				else if(pcrx->getStruxType() == PTX_SectionCell)
				{
					sCRArgs += "cell ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndCell)
				{
					sCRArgs += "endcell ";
				}
				else if(pcrx->getStruxType() == PTX_SectionFrame)
				{
					sCRArgs+= "frame ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndFrame)
				{
					sCRArgs += "endframe";
				}
			}
			break;
		case PX_ChangeRecord::PXT_DeleteStrux:
			sCRType = "PXT_DeleteStrux";
			iAdjust = -1;
			{
				const PX_ChangeRecord_Strux * pcrx = static_cast<const PX_ChangeRecord_Strux *> (pcr);
				if(pcrx->getStruxType() == PTX_Block)
				{
					sCRArgs += "p";
				}
				else if(pcrx->getStruxType() == PTX_Section)
				{
					sCRArgs += "section";
				}
				else if(pcrx->getStruxType() == PTX_SectionFootnote)
				{
					sCRArgs += "footnote";
				}
				else if(pcrx->getStruxType() == PTX_EndFootnote)
				{
					sCRArgs += "endfootnote";
				}
				else if(pcrx->getStruxType() == PTX_SectionTOC)
				{
					sCRArgs += "toc";
				}
				else if(pcrx->getStruxType() == PTX_EndTOC)
				{
					sCRArgs += "endtoc";
				}
				else if(pcrx->getStruxType() == PTX_SectionEndnote)
				{
					sCRArgs += "endnote";
				}
				else if(pcrx->getStruxType() == PTX_EndEndnote)
				{
					sCRArgs += "endendnote";
				}
				else if(pcrx->getStruxType() == PTX_SectionTable)
				{
					sCRArgs += "table";
				}
				else if(pcrx->getStruxType() == PTX_EndTable)
				{
					sCRArgs += "endtable";
				}
				else if(pcrx->getStruxType() == PTX_SectionCell)
				{
					sCRArgs += "cell";
				}
				else if(pcrx->getStruxType() == PTX_EndCell)
				{
					sCRArgs += "endcell";
				}
				else if(pcrx->getStruxType() == PTX_SectionFrame)
				{
					sCRArgs += "frame";
				}
				else if(pcrx->getStruxType() == PTX_EndFrame)
				{
					sCRArgs += "endframe";
				}
			}
			break;
		case PX_ChangeRecord::PXT_ChangeStrux:
			sCRType = "PXT_ChangeStrux";
			{
				const PX_ChangeRecord_StruxChange * pcrx = static_cast<const PX_ChangeRecord_StruxChange *> (pcr);
				if(pcrx->getStruxType() == PTX_Block)
				{
					sCRArgs += "p";
					if(index != m_iBlockIndex)
					{
						m_iBlockIndex = index;
					}
					sCRArgs += " ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_Section)
				{
					sCRArgs += "section";
					if(index != m_iSectionIndex)
					{
						m_iSectionIndex = index;
					}
					sCRArgs += " ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_SectionFootnote)
				{
					sCRArgs += "footnote ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndFootnote)
				{
					sCRArgs += "endfootnote";
				}
				else if(pcrx->getStruxType() == PTX_SectionTOC)
				{
					sCRArgs += "toc ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndTOC)
				{
					sCRArgs += "endtoc";
				}
				else if(pcrx->getStruxType() == PTX_SectionEndnote)
				{
					sCRArgs += "endnote ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndEndnote)
				{
					sCRArgs += "endendnote";
				}
				else if(pcrx->getStruxType() == PTX_SectionTable)
				{
					sCRArgs += "table ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndTable)
				{
					sCRArgs += "endtable";
				}
				else if(pcrx->getStruxType() == PTX_SectionCell)
				{
					sCRArgs += "cell ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndCell)
				{
					sCRArgs += "endcell";
				}
				else if(pcrx->getStruxType() == PTX_SectionFrame)
				{
					sCRArgs += "frame ";
					mapAttributesToString(index, sCRArgs);
				}
				else if(pcrx->getStruxType() == PTX_EndFrame)
				{
					sCRArgs += "endframe";
				}
			}
			break;
		case PX_ChangeRecord::PXT_InsertObject:
			sCRType = "PXT_InsertObject";
			iAdjust =  1;
			{
				const PX_ChangeRecord_Object * pcro = static_cast<const PX_ChangeRecord_Object *>(pcr);
				if(pcro->getObjectType() == PTO_Image)
				{
					sCRArgs += "image ";
				}
				else if(pcro->getObjectType() == PTO_Field)
				{
					sCRArgs += "field ";
				}
				else if(pcro->getObjectType() == PTO_Bookmark)
				{
					sCRArgs += "bookmark ";
				}
				else if(pcro->getObjectType() == PTO_Hyperlink)
				{
					sCRArgs += "hyperlink ";
				}
				else if(pcro->getObjectType() == PTO_Math)
				{
					sCRArgs += "math ";
				}
				else if(pcro->getObjectType() == PTO_Embed)
				{
					sCRArgs += "embed ";
				}
				mapAttributesToString(index, sCRArgs);
				UT_DEBUGMSG(("Insert Object attribs |%s| \n",sCRArgs.utf8_str()));
			}
			break;
		case PX_ChangeRecord::PXT_DeleteObject:
			sCRType = "PXT_DeleteObject";
			iAdjust = -1;
			{
				const PX_ChangeRecord_Object * pcro = static_cast<const PX_ChangeRecord_Object *>(pcr);
				if(pcro->getObjectType() == PTO_Image)
				{
					sCRArgs += "image";
				}
				else if(pcro->getObjectType() == PTO_Field)
				{
					sCRArgs += "field";
				}
				else if(pcro->getObjectType() == PTO_Bookmark)
				{
					sCRArgs += "bookmark";
				}
				else if(pcro->getObjectType() == PTO_Hyperlink)
				{
					sCRArgs += "hyperlink";
				}
				else if(pcro->getObjectType() == PTO_Math)
				{
					sCRArgs += "math";
				}
				else if(pcro->getObjectType() == PTO_Embed)
				{
					sCRArgs += "embed";
				}
			}
			break;
		case PX_ChangeRecord::PXT_ChangeObject:
			sCRType = "PXT_ChangeObject";
			{
				const PX_ChangeRecord_ObjectChange * pcro = static_cast<const PX_ChangeRecord_ObjectChange *>(pcr);
				if(pcro->getObjectType() == PTO_Image)
				{
					sCRArgs += "image ";
				}
				else if(pcro->getObjectType() == PTO_Field)
				{
					sCRArgs += "field ";
				}
				else if(pcro->getObjectType() == PTO_Bookmark)
				{
					sCRArgs += "bookmark ";
				}
				else if(pcro->getObjectType() == PTO_Hyperlink)
				{
					sCRArgs += "hyperlink ";
				}
				else if(pcro->getObjectType() == PTO_Math)
				{
					sCRArgs += "math ";
				}
				else if(pcro->getObjectType() == PTO_Embed)
				{
					sCRArgs += "embed ";
				}
				mapAttributesToString(index, sCRArgs);
			}
			break;
		case PX_ChangeRecord::PXT_InsertFmtMark:
			sCRType = "PXT_InsertFmtMark";
			sCRArgs += "span ";
			mapAttributesToString(index, sCRArgs);
			break;
		case PX_ChangeRecord::PXT_DeleteFmtMark:
			sCRType = "PXT_DeleteFmtMark";
			sCRArgs += "span";
			break; 
		case PX_ChangeRecord::PXT_ChangeFmtMark:
			sCRType = "PXT_ChangeFmtMark";
			sCRArgs += "span ";
			mapAttributesToString(index, sCRArgs);
			break;
		case PX_ChangeRecord::PXT_ChangePoint:
			sCRType = "PXT_ChangePoint";
			break; 
		case PX_ChangeRecord::PXT_ListUpdate:
			sCRType = "PXT_ListUpdate";
			break; 
		case PX_ChangeRecord::PXT_StopList:
			sCRType = "PXT_StopList";
			break; 
		case PX_ChangeRecord::PXT_UpdateField:
			sCRType = "PXT_UpdateField";
			break;
		case PX_ChangeRecord::PXT_RemoveList:
			sCRType = "PXT_RemoveList";
			break;
		case PX_ChangeRecord::PXT_UpdateLayout:
			sCRType = "PXT_UpdateLayout";
			break;
		case PX_ChangeRecord::PXT_CreateDataItem:
			sCRType = "PXT_CreateDataItem";
			{
			     sCRArgs += "<data ";
			     mapAttributesToString(index, sCRArgs);
			     const PP_AttrProp * pAP = NULL;
			     PT_AttrPropIndex indexAP = static_cast<PT_AttrPropIndex>(index);
			     bool b = m_pDoc->getAttrProp(indexAP, &pAP);
			     UT_DEBUGMSG(("dataItem Index = %d pAP %x \n",indexAP,pAP));
			     UT_return_val_if_fail(b,b);
			     const gchar * pszDataName = NULL;
			     (pAP)->getAttribute(PT_DATAITEM_ATTRIBUTE_NAME,pszDataName);
			     if(pszDataName == NULL)
			     {
				 UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
				 break;
			     }
			     const UT_ByteBuf * pBuf=NULL;
			     const void *pToken = NULL;
			     void *pHandle = NULL;
			     m_pDoc->getDataItemDataByName(pszDataName,&pBuf,&pToken,&pHandle);
			     UT_ByteBuf * pB64 = new UT_ByteBuf();
			     UT_Base64Encode(pB64,pBuf);

			     UT_uint32 iLen = pB64->getLength();
				
			     const char * pChars = reinterpret_cast<const char *>(pB64->getPointer(0));
			     UT_UTF8String sText;
			     sText.append(pChars,iLen);
			     sText = sText.escapeXML();
			     iLen= sText.length();
			     UT_UTF8String sPos;
			     UT_UTF8String_sprintf(sPos,"%d",iLen);

			     sCRArgs += " length=\"";
			     sCRArgs += sPos;
			     sCRArgs += "\">";
			     sCRArgs += sText;
			     sCRArgs += "</data>";
			     iAdjust = 0;
			     delete pB64;
			}

			break;
		default:
			break;
	}
	
	// WARNING WARNING: this is 100% unmaintainable... rewrite this!
	
	// wow this is ugly: only in the PXT_InsertSpan and PXT_CreateDataItem
	// create their own opening and closing tags
	if (sCRArgs.length() > 0 &&
		(pcr->getType() != PX_ChangeRecord::PXT_InsertSpan &&
		pcr->getType() != PX_ChangeRecord::PXT_CreateDataItem))
	{
		UT_UTF8String tmp = "<";
		tmp += sCRArgs;
		tmp += "/>";
		sCRArgs = tmp;
	}
}


/*!
 * This method converts a change record into a text structure that can be 
 * either stored ina file to record actions taken on AbiWord or sent over the
 * internet to remove instance of AbiWord where it's contents are translated
 * back into actions on the remote AbiWord.
 */
bool  ABI_Collab_Export::buildPacket(bool bPopulate,
				     const PX_ChangeRecord * pcr,
				     UT_UTF8String & sPacket)
{
	UT_UTF8String sType, sArgs;
	UT_sint32 dPos = pcr->getPosition();
	UT_sint32 iAdjust = 0;
	mapChangeRecordToString(pcr, sType, sArgs, iAdjust);
	if(!m_pAbiCollab->isExportMasked())
	{
	    ChangeAdjust * pChange = new ChangeAdjust(dPos,iAdjust,pcr->getCRNumber());
	    m_vecAdjusts.addItem(pChange);
	}
	sPacket = "";
	
	// Open the ChangeRecord
	if(!bPopulate)
	{
		sPacket += "<cr type=\"";
	}
	else
	{
		sPacket += "<pop type=\"";
	}
	sPacket += sType;
	sPacket += "\"";

	// docUUID
	UT_UTF8String sTrans;
	sTrans = m_pDoc->getOrigDocUUIDString();
	sPacket += " docUUID=\"";
	sPacket += sTrans;
	sPacket += "\"";

	// myUUID
	sTrans = pcr->getMyUUID();
	sPacket += " myUUID=\"";
	sPacket += sTrans;
	sPacket += "\"";

	// Change Record Number
 
	UT_UTF8String_sprintf(sTrans,"%d",pcr->getCRNumber());
	sPacket += " CRNum=\"";
	sPacket += sTrans;
	sPacket += "\"";

	// Last CRNum received
    ABI_Collab_Import * pImport = m_pAbiCollab->getImport();
    const UT_GenericVector<ChangeAdjust *> * pImpAdjusts = pImport->getAdjusts();
	UT_sint32 iLastCR = 0;
	UT_sint32 iNumImps = pImpAdjusts->getItemCount();
	if(iNumImps > 0)
	{
	    iLastCR = pImpAdjusts->getNthItem(iNumImps-1)->m_iCRNumber;
	}
	UT_UTF8String_sprintf(sTrans,"%d",iLastCR);
	sPacket += " IMPCRNum=\"";
	sPacket += sTrans;
	sPacket += "\"";
	
	// GLOB type
	if(pcr->getType() == PX_ChangeRecord::PXT_GlobMarker)
	{
		const PX_ChangeRecord_Glob * pcrg = static_cast<const PX_ChangeRecord_Glob *>(pcr);
		UT_UTF8String sGlob;
		UT_UTF8String_sprintf(sGlob,"%d",pcrg->getFlags());
		sPacket += " globtype=\"";
		sPacket += sGlob;
		sPacket += "\"";
		
	}

	// Position
	UT_UTF8String sPos;
	UT_UTF8String_sprintf(sPos,"%d",pcr->getPosition());
	sPacket += " pos=\"";
	sPacket += sPos;
	sPacket += "\"";
	sPacket += ">";

	// Output the ChangeRecord arguments
	if(sArgs.size() > 0)
	{
		sPacket += sArgs;
	}
	
	// Close the ChangeRecord
	if(!bPopulate)
	{
		sPacket += "</cr>";
	}
	else
	{
		sPacket += "</pop>";
	}
	
	return true;
}

/*!
 * This method extracts all the attributes (including the property string
 * from the index provides and maps them to a utf-8 string.
 * The result can be placed inside an XML name space. 
 */
bool  ABI_Collab_Export::mapAttributesToString(UT_sint32 indx, UT_UTF8String & sAttribs)
{
	const PP_AttrProp * pAP = NULL;
	PT_AttrPropIndex indexAP = static_cast<PT_AttrPropIndex>(indx);
	bool b = m_pDoc->getAttrProp(indexAP, &pAP);
	UT_return_val_if_fail(b,b);
	UT_sint32 nAtts = static_cast<UT_sint32>(pAP->getAttributeCount());
	UT_sint32 i = 0;
	const gchar * szName = NULL;
	const gchar * szVal = NULL;
	for(i=0; i<nAtts;i++)
	{
		pAP->getNthAttribute(i,szName,szVal);
		if(strstr(sAttribs.utf8_str(),szName) == NULL)
		{
			UT_DEBUGMSG(("Export: %d Name |%s| Val |%s| \n",i,szName,szVal));
			sAttribs += szName;
			sAttribs += "=\"";
			sAttribs += szVal;
			sAttribs += "\"";
			sAttribs += " ";
		}
	}
	UT_sint32 nProps = static_cast<UT_sint32>(pAP->getPropertyCount());
	if(nProps > 0)
	{
		sAttribs += " props=\"";
		for(i=0; i<nProps;i++)
		{
			pAP->getNthProperty(i,szName,szVal);
			sAttribs += szName;
			sAttribs +=":";
			sAttribs +=szVal;
			if(i+1 <nProps)
			sAttribs +=";";
		}
		sAttribs += "\"";
	}
}


/*!
 * Implements the "populate" method of the document listener class.
 * It takes the supplied change record and makes an XML-like representation
 * of it. Eventually these can either be stored in a file or sent over the
 * internet to a remote AbiWord where it can be translated back.
 */
bool ABI_Collab_Export::populate(PL_StruxFmtHandle sfh, const PX_ChangeRecord * pcr)
{
	UT_UTF8String sPacket;
	sPacket.clear();
	buildPacket(true,pcr,sPacket);
	m_pAbiCollab->push(sPacket);
	return true;
}

/*!
 * Implements the "populateStrux" method of the document listener class.
 * It takes the supplied change record and makes an XML-like representation
 * of it. Eventually these can either be stored in a file or sent over the
 * internet to a remote AbiWord where it can be translated back.
 */
bool ABI_Collab_Export::populateStrux(PL_StruxDocHandle sdh,
				      const PX_ChangeRecord * pcr,
				      PL_StruxFmtHandle * psfh)
{
	*psfh = sdh;
	UT_UTF8String sPacket;
	sPacket.clear();
	buildPacket(true,pcr,sPacket);
	m_pAbiCollab->push(sPacket);
	return true;
}

/*!
 * Returns true if the stop flag matches the start.
 */
bool ABI_Collab_Export::compareGLOBFlags(UT_Byte istart, UT_Byte istop)
{
	if((istart == PX_ChangeRecord_Glob::PXF_UserAtomicStart) &&
	   (istop == PX_ChangeRecord_Glob::PXF_UserAtomicEnd))
		return true;
	if((istart == PX_ChangeRecord_Glob::PXF_MultiStepStart) &&
	   (PX_ChangeRecord_Glob::PXF_MultiStepEnd))
		return true;
	return false;
}
/*!
 * Implements the "change" method of the document listener class.
 * It takes the supplied change record and makes an XML-like representation
 * of it. Eventually these can either be stored in a file or sent over the
 * internet to a remote AbiWord where it can be translated back.
 */
bool ABI_Collab_Export::change(PL_StruxFmtHandle sfh,
				const PX_ChangeRecord * pcr)
{
	UT_UTF8String sPacket;
	sPacket.clear();
	buildPacket(false,pcr,sPacket);
	if(pcr->getType() == PX_ChangeRecord::PXT_GlobMarker)
	{
		const PX_ChangeRecord_Glob * pcrg = static_cast<const PX_ChangeRecord_Glob *>(pcr);
		UT_DEBUGMSG(("Got a GLOB flag %x CurFlag %x \n",pcrg->getFlags(),m_iGlobFlag));
	}
	if(!m_bDoingGlob && (pcr->getType() != PX_ChangeRecord::PXT_GlobMarker))
	{
		m_pAbiCollab->push(sPacket);
		return true;
	}
	else if(m_bDoingGlob && (pcr->getType() != PX_ChangeRecord::PXT_GlobMarker))
	{
		m_sPacket += sPacket;
		xxx_UT_DEBUGMSG(("Packet grows to |%s| \n",m_sPacket.utf8_str()));
		return true;
	}
	else if(m_bDoingGlob && (pcr->getType() == PX_ChangeRecord::PXT_GlobMarker))
	{
		const PX_ChangeRecord_Glob * pcrg = static_cast<const PX_ChangeRecord_Glob *>(pcr);
		xxx_UT_DEBUGMSG(("Got a GLOB flag %x Orig %x \n",pcrg->getFlags(),m_iGlobFlag));
		if(compareGLOBFlags(m_iGlobFlag,pcrg->getFlags()))
		{
			m_sPacket += sPacket;
			xxx_UT_DEBUGMSG(("Finished GLOB. Sending packet \n"));
			m_bDoingGlob = false;
			m_sPacket += "</glob>";
			m_pAbiCollab->push(m_sPacket);
			return true;
		}
		if(pcrg->getFlags()==PX_ChangeRecord_Glob::PXF_UserAtomicStart)
		{
			UT_DEBUGMSG(("New GLOB begin before end! Recover.. \n"));
			UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
			m_sPacket += sPacket;
			m_sPacket += "</glob>";
			m_pAbiCollab->push(m_sPacket);
			m_sPacket = "<glob>";
			m_iGlobFlag = pcrg->getFlags();
		}
		m_sPacket += sPacket;
		xxx_UT_DEBUGMSG(("Packet grows to |%s| \n",m_sPacket.utf8_str()));
		return true;
	}
	m_bDoingGlob = true;
	m_iGlobFlag = static_cast<const PX_ChangeRecord_Glob *>(pcr)->getFlags();
	xxx_UT_DEBUGMSG(("Original GLOB flag %x \n",m_iGlobFlag));
	m_sPacket = "<glob>";
	m_sPacket += sPacket;
	xxx_UT_DEBUGMSG(("Initial Packet is |%s| \n",m_sPacket.utf8_str()));
	return true;
}


/*!
 * Implements the "insertStrux" method of the document listener class.
 * It takes the supplied change record and makes an XML-like representation
 * of it. Eventually these can either be stored in a file or sent over the
 * internet to a remote AbiWord where it can be translated back.
 */
bool ABI_Collab_Export::insertStrux(PL_StruxFmtHandle sfh,
									const PX_ChangeRecord * pcr,
									PL_StruxDocHandle sdh,
									PL_ListenerId lid,
									void (* pfnBindHandles)(PL_StruxDocHandle sdhNew,
															PL_ListenerId lid,
															PL_StruxFmtHandle sfhNew))
{
	
	UT_UTF8String sType;
	if(pfnBindHandles)
	{
		PL_StruxFmtHandle sfhNew = static_cast<PL_StruxFmtHandle>(this);
		pfnBindHandles(sdh,lid,sfhNew);
	}
	UT_UTF8String sPacket;
	sPacket.clear();
	buildPacket(false,pcr,sPacket);
	if(m_bDoingGlob)
	{
		m_sPacket += sPacket;
		xxx_UT_DEBUGMSG(("Packet grows to |%s| \n",m_sPacket.utf8_str()));
	}
	else
	{
		m_pAbiCollab->push(sPacket);
	}
	return true;
}

/*!
 * Don't know if we need this method for abiCollab
 */
void  ABI_Collab_Export::deferNotifications(void)
{
}

/*!
 * Don't know if we need this method for abiCollab
 */
void ABI_Collab_Export::processDeferredNotifications(void)
{
}

/*!
 * Implements the signal() method of the Document listener class.
 */
bool ABI_Collab_Export::signal(UT_uint32 iSignal)
{
	UT_DEBUGMSG(("FIXME FIXME FIXME, DON'T SEND SIGNALS DIRECTLY FROM ABI_Collab_Export::signal()!"));
	
	SignalSessionPacket sp("FIXME FIXME FIXME, DON'T SEND SIGNALS FROM HERE!!!!", iSignal);
	const UT_UTF8String* psPacket = sp.serialize();
	if (!psPacket)
		return false;
	if(!m_bDoingGlob)
	{
		m_pAbiCollab->push(*psPacket);
	}
	else
	{
		m_sPacket += *psPacket;
		xxx_UT_DEBUGMSG(("Packet grows to |%s| \n",m_sPacket.utf8_str()));
	}
	return true;
}

void ABI_Collab_Export::clearGLOB(void)
{
	m_bDoingGlob = false;
	m_iGlobFlag = 0;
	m_sPacket.clear();
	m_iGlobCounter = 0;
}

/*!
 * This virtual method is called from the AbiWord main tree upon doing a replace document with an attached
 * AbiCollab_Export connected to the document.
 *
 * Note: this is a really weird signal, coming from a PD_Document
 * Note: If anything, a Frame should emit this signal to its listeners
 */
void ABI_Collab_Export::setNewDocument(PD_Document * pDoc)
{
	UT_DEBUGMSG(("ABI_Collab_Export::setNewDocument() - ignored\n"));
	// we are connected to a session, we can't just replace the document!
	UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
	
	// TODO: inform the session manager to kill off this session, as it has seriously gone bad
	// ...
}

/*!
 * This virtual method is called if the attached document is deleted with an attached
 * AbiCollab_Export connected to the document.
 */
void ABI_Collab_Export::removeDocument(void)
{
	UT_DEBUGMSG(("ABI_Collab_Export::removeDocument()\n"));

	// inform the session manager that this session is being (forcefully) closed
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	UT_return_if_fail(pManager);
	
	pManager->disconnectSession(m_pAbiCollab);
	// NOTE: don't do anything after this line anymore, as this will self-destruct us!
	pManager->destroySession(m_pAbiCollab);
}
