/* Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef __ACCOUNTHANDLER_H__
#define __ACCOUNTHANDLER_H__

#include <map>
#include <string>
#include <list>
#include <vector>
#include "ut_types.h"
#include "ut_string_class.h"
#include <handlers/xp/Buddy.h>
#include "ut_vector.h"
#include "DocHandle.h"
#include <xp/AbiCollab_Packet.h>
#include <handlers/xp/EventListener.h>

class AbiCollab;

using std::string;
using std::map;
using std::vector;

typedef AccountHandler* (*AccountHandlerConstructor)();

typedef map<string, string> PropertyMap; 

ABI_EXPORT class AccountHandler : public EventListener
{
public:
	AccountHandler() {}
	virtual ~AccountHandler() {}

	// housekeeping
	virtual UT_UTF8String					getDescription() = 0;	
	virtual UT_UTF8String					getDisplayType() = 0;
	virtual UT_UTF8String					getStorageType() = 0;	
	
	void									addProperty(const string& key, const string& value)
		{ m_properties.insert(PropertyMap::value_type(key, value)); }
	const string							getProperty(const string& key);
	PropertyMap&							getProperties()
		{ return m_properties; }

	// dialog management 
	virtual void							embedDialogWidgets(void* pEmbeddingParent) = 0;
	virtual void							removeDialogWidgets(void* pEmbeddingParent) = 0;
	virtual void							storeProperties() = 0;

	// connection management
	virtual UT_sint32						connect() = 0;
	virtual bool							disconnect() = 0;
	virtual bool							isOnline() = 0;
	virtual bool							autoConnect();

	// packet management
	// TODO: we should only send Packet's, not raw strings
	bool									send(AbiCollab* pSession, const UT_UTF8String& packet, PacketType packetType);
	bool									send(AbiCollab* pSession, const UT_UTF8String& packet, PacketType packetType, const Buddy& buddy);
	
	// user management
	void									addBuddy(Buddy* buddy);
	const UT_GenericVector<Buddy*>&	 		getBuddies() const
		{ return m_vecBuddies; }

	Buddy*									getBuddy(const UT_UTF8String& name);
	virtual Buddy*							constructBuddy(const PropertyMap& vProps) = 0;
	
	bool getCanOffer()
		{ return m_bCanOffer; }

	void setOffering(bool bCanOffer)
		{ m_bCanOffer = bCanOffer; }
		
	// session management
	void									getSessionsAsync();
	virtual void							getSessionsAsync(const Buddy& buddy) = 0;
	virtual void							joinSessionAsync(const Buddy& buddy, DocHandle& docHandle) = 0;

	// signal management
	virtual void							signal(const Event& event, const Buddy* pSource);
	
protected:

	// packet management
	virtual bool							_send(const UT_UTF8String& packet) = 0;
	virtual bool							_send(const UT_UTF8String& packet, const Buddy& buddy) = 0;
	bool									_decoratePacket(const AbiCollab* pSession, const UT_UTF8String& packet, const PacketType packetType, UT_UTF8String& sDecoratedPacket);

	// bad bad, protected variables are bad
	PropertyMap								m_properties;

private:
	bool									m_bCanOffer;
	UT_GenericVector<Buddy*>				m_vecBuddies;
};

#endif /* ACCOUNTHANDLER_H */
