/* Copyright (C) 2006,2007 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <xp/AbiCollab_Plugin.h>
#include <xp/AbiCollab.h>
#include <xp/AbiCollabSessionManager.h>
#include "AccountEvent.h"
#include "AccountHandler.h"

const string AccountHandler::getProperty(const string& key)
{ 
	PropertyMap::iterator pos = m_properties.find(key);
	if (pos != m_properties.end())
		return pos->second;
	else
		return "";
}

bool AccountHandler::autoConnect()
{
	const std::string autoconnect = getProperty("autoconnect");
	return strcmp(autoconnect.c_str(), "true") == 0;
}

/*!
 Send a collaboration packet to all registered buddies
 */ 
bool AccountHandler::send(AbiCollab* pSession, const UT_UTF8String& packet, PacketType packetType)
{
	UT_UTF8String sDecoratedPacket;
	if (_decoratePacket(pSession, packet, packetType, sDecoratedPacket))
	{
		return _send(sDecoratedPacket);
	}
	return false;
}

/*!
  Send a collaboration packet to a buddy
 */	
bool AccountHandler::send(AbiCollab* pSession, const UT_UTF8String& packet, PacketType packetType, const Buddy& buddy)
{
	UT_UTF8String sDecoratedPacket;
	if (_decoratePacket(pSession, packet, packetType, sDecoratedPacket))
	{
		return _send(sDecoratedPacket, buddy);
	}
	return false;
}

void AccountHandler::addBuddy(Buddy* buddy)
{
	m_vecBuddies.push_back(buddy);
	
	// signal all listeners we have a new buddy
	AccountAddBuddyEvent event;
	// TODO: fill the event
	AbiCollabSessionManager::getManager()->signal(event);
}

Buddy* AccountHandler::getBuddy(const UT_UTF8String& name)
{
	for (UT_sint32 i = 0; i < m_vecBuddies.getItemCount(); i++)
	{
		Buddy* pBuddy = m_vecBuddies.getNthItem(i);
		if (pBuddy->getName() == name)
			return pBuddy;
	}		
	UT_DEBUGMSG(("Getting buddy (%s) failed\n", name.utf8_str()));
	return 0;
}
		
void AccountHandler::getSessionsAsync()
{
	for (UT_sint32 i = 0; i < m_vecBuddies.getItemCount(); i++)
	{
		const Buddy* pBuddy = m_vecBuddies.getNthItem(i);
		getSessionsAsync(*pBuddy);
	}
}

void AccountHandler::signal(const Event& event, const Buddy* pSource)
{
	UT_DEBUGMSG(("AccountHandler::signal()\n"));

	// broadcast this event over our network (if applicable for each message type)
	EventPacket* pPacket = event.constructPacket();
	if (pPacket)
	{
		UT_GenericVector<Buddy*> vRecipients = 
			(event.isBroadcast() ? getBuddies() : event.getRecipients());
		
		for (UT_sint32 i = 0; i < vRecipients.getItemCount(); i++)
		{
			Buddy* pRecipient = vRecipients.getNthItem(i);
			if (pRecipient)
			{
				if (!pSource || 
					(pSource && pRecipient->getName() != pSource->getName())
					)
				{
					send(0, pPacket->serialize(), pPacket->getType(), *pRecipient);
				}
				else
				{
					// the event came originated at this buddy, so make sure not to send it
					// back to him, as it would result in a broadcast storm and
					// kill the network really fast
				}
			}
		}
	}
}

bool AccountHandler::_decoratePacket(const AbiCollab* pSession, const UT_UTF8String& packet, const PacketType packetType, UT_UTF8String& sDecoratedPacket)
{
	// TODO: merge this code with ::send(pSession, packet)
	if (packetType == PT_Session && pSession == 0)
	{
		UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
		return false;
	}
	
	sDecoratedPacket += "<abicollab protocol=\"";
	sDecoratedPacket += ABICOLLAB_PROTOCOL_VERSION;
	sDecoratedPacket += "\">";
	
	switch (packetType)
	{
		case PT_Session:
			sDecoratedPacket += "<session id=\"";
			sDecoratedPacket += pSession->getSessionId();
			sDecoratedPacket += "\">";
			sDecoratedPacket += packet;
			sDecoratedPacket += "</session>";
			break;
		case PT_Event:
			sDecoratedPacket += "<event>";
			sDecoratedPacket += packet;
			sDecoratedPacket += "</event>";
			break;
		case PT_Handler:
			sDecoratedPacket += "<handler>";
			sDecoratedPacket += packet;
			sDecoratedPacket += "</handler>";
			break;
		default:
			UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
			break;
	}
	sDecoratedPacket += "</abicollab>";
	
	return true;
}
