/*
 * AbiCollab - Code to enable the modification of remote documents.
 * Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
 
#ifndef ABICOLLAB_PACKET_H
#define ABICOLLAB_PACKET_H

#include <libxml/xmlreader.h>

#include "ut_types.h"
#include "ut_string_class.h"
#include "px_ChangeRecord.h"
#include "ut_stack.h"

class Buddy;

enum PacketType
{
	PT_Session = 0,
	PT_Event,
	PT_Handler
};
	
enum PacketSessionType
{
	PACKET_CHANGERECORD_POPULATE,
	PACKET_CHANGERECORD_CR,
	PACKET_SIGNAL,
	PACKET_GLOB
};

enum PacketEventType
{
	PTE_AccountAddBuddyRequest,

	PTE_StartSession,
	PTE_JoinSession,
	PTE_DisjoinSession,
	PTE_CloseSession,
};

struct RawPacket
{
	Buddy* buddy; // TODO: free the buddy somewhere, or make it not a pointer (please do the latter) - MARCM
	UT_UTF8String packet;
};

class PX_ChangeRecord;
class SessionPacket;

class ABI_EXPORT SessionPacketFactory
{
public:
	SessionPacketFactory(const RawPacket& data);
	SessionPacketFactory(xmlNode* sessionNode);
	virtual ~SessionPacketFactory(void);
	
	SessionPacket *		getNextPacket(void);
	bool				getChangeRecordType(const UT_UTF8String& sType, 
								PX_ChangeRecord::PXType & crType);
	UT_uint32			getSignalType(const UT_UTF8String& sType);

private:
	SessionPacket*		_getNextPacket(xmlNode* pPacketNode, const UT_UTF8String& sSessionId);
	SessionPacket*		_getNextSessionPacket(xmlNode* pSessionNode);
	
	xmlDocPtr			m_pDocReader;
	xmlNode*			m_pCurNode;
};

/*************************************************************
 * Packets                                                   *
 *************************************************************/

class ABI_EXPORT Packet
{
public:
	Packet();
	virtual ~Packet();

	virtual PacketType				getType() const = 0;

	bool							fillPTAttributes(xmlNode * node);
	bool							fillPropsFromString(UT_UTF8String& sProps); 
	char *							getAttribute(char * szName) const;
		
	// NOTE: is an UT_UTF8String the best format for this?
	virtual const UT_UTF8String		serialize() = 0;

protected:
	XML_Char**						m_szAtts;
	XML_Char**						m_szProps;
	void							_freeProps(); 
	void							_freeAtts();
};


/*************************************************************
 * SessionPackets                                            *
 *************************************************************/

class ABI_EXPORT SessionPacket : public Packet
{
public:
	SessionPacket(const UT_UTF8String& sSessionId);
	virtual ~SessionPacket();
	
	virtual PacketType getType() const
		{ return PT_Session; }

	virtual PacketSessionType getSessionType() const = 0;

	const UT_UTF8String& getSessionId()
		{ return m_sSessionId; }

	void setSessionId(const UT_UTF8String& sessionId)
		{ m_sSessionId = sessionId; }

	// FIXME: This function should be implemented by the specific subclasses
	const UT_UTF8String				serialize()
		{ return "TODO: IMPLEMENT ME"; }
		
private:
	UT_UTF8String			m_sSessionId;	
};

class ABI_EXPORT ChangeRecordSessionPacket : public SessionPacket
{
public:
	ChangeRecordSessionPacket(
			const UT_UTF8String& sSessionId,
			PX_ChangeRecord::PXType cType, 
			const UT_UTF8String& sDocUUID, 
			const UT_UTF8String& sMyUUID, 
			int iCRNum, 
			int iImpCRNum, 
			int iPos,
			UT_Byte iGLOB,
			PacketSessionType crType);
	
	virtual ~ChangeRecordSessionPacket(void);

	virtual PacketSessionType getSessionType() const
		{ return m_crType; }

	void setSpanProps(size_t iLength, const char* value);

	PX_ChangeRecord::PXType getPXType() const
		{ return m_cType; }

	const UT_UTF8String& getDocUUID() const
		{ return m_sDocUUID; }
	
	const UT_UTF8String& getMyUUID() const
		{ return m_sMyUUID; }
		
	int getCRNum() const
		{ return m_iCRNum; }

	int getImpCRNum(void) const
		{ return  m_iImpCRNum; }

	int getPos() const
		{ return m_iPos; }

	UT_Byte getGLOBType() const
		{ return m_iGLOBType;}

	const UT_UTF8String& getFrag() const
		{ return m_sFrag; }	
		
	size_t getLength() const
		{ return m_iLength; }
		
	XML_Char** getProps() const
		{ return m_szProps; }
		
	XML_Char** getAtts() const
		{ return m_szAtts; }
		
	const UT_UTF8String& getValue() const
		{ return m_sValue; }

	void setFrag(const xmlChar * szFrag)
		{ m_sFrag = reinterpret_cast<const char *>(szFrag); }

private:
	PX_ChangeRecord::PXType		m_cType;
	UT_UTF8String				m_sDocUUID;
	UT_UTF8String				m_sMyUUID;
	int							m_iCRNum;
	int							m_iPos;
	int							m_iImpCRNum;

	UT_UTF8String				m_sFrag;
	size_t						m_iLength;

	UT_UTF8String				m_sValue;
	UT_Byte						m_iGLOBType;

	PacketSessionType			m_crType;
};

class ABI_EXPORT SignalSessionPacket : public SessionPacket
{
public:
	SignalSessionPacket(const UT_UTF8String& sSessionId, UT_uint32 iSignal);

	virtual const UT_UTF8String* serialize() const;

	virtual PacketSessionType getSessionType() const
		{ return PACKET_SIGNAL; }
	
	UT_uint32 getSignalType() const
		{ return m_iSignal; }
		
private:
	UT_uint32	m_iSignal;
};

class ABI_EXPORT GlobSessionPacket : public SessionPacket
{
public:
	GlobSessionPacket(const UT_UTF8String& sessionId);
	virtual ~GlobSessionPacket();

	virtual PacketSessionType getSessionType() const
		{ return PACKET_GLOB; }

	void addPacket(SessionPacket* pPacket)
		{ m_pPackets.push_back(pPacket); }

	const UT_GenericVector<SessionPacket*>& getPackets() const
		{ return m_pPackets; }
		
private:
	UT_GenericVector<SessionPacket*>		m_pPackets;
};

/*************************************************************
 * HandlerPackets                                            *
 *************************************************************/

// ...

#endif /* ABICOLLAB_PACKET_H */
