/* Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "xap_App.h"
#include "ap_UnixApp.h"
#include "xap_UnixApp.h"
#include "xap_Frame.h"
#include "xap_UnixDialogHelper.h"
#include "ut_string_class.h"
#include <xp/AbiCollabSessionManager.h>
#include <handlers/xp/Event.h>
#include <handlers/xp/AccountEvent.h>
#include <handlers/xmpp/xp/XMPPBuddy.h>

#include "ap_UnixDialog_CollaborationJoin.h"

enum
{
	DESCRIPTION_COLUMN = 0,
	JOINED_COLUMN,
	DOCHANDLE_COLUMN,
	BUDDY_COLUMN,
	VISIBLE_COLUMN,
	NUM_COLUMNS
};

static void s_add_buddy_clicked(GtkWidget * wid, AP_UnixDialog_CollaborationJoin * dlg)
{
	dlg->eventAddBuddy();
}

static void s_refresh_clicked(GtkWidget * wid, AP_UnixDialog_CollaborationJoin * dlg)
{
	dlg->eventRefresh();
}

static void joined_toggled (GtkCellRendererToggle *cell,
	      gchar                 *path_str,
	      gpointer               data)
{
	AP_UnixDialog_CollaborationJoin* pDlg = static_cast<AP_UnixDialog_CollaborationJoin*>(data);
	GtkTreeModel *model = GTK_TREE_MODEL(pDlg->getModel());
	GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
	GtkTreeIter iter;
	
	gboolean joined;
	gpointer doc_handle;
	gpointer buddy;	

	// get the toggled state
	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, JOINED_COLUMN, &joined, -1);
	gtk_tree_model_get (model, &iter, DOCHANDLE_COLUMN, &doc_handle, -1);
	gtk_tree_model_get (model, &iter, BUDDY_COLUMN, &buddy, -1);	

	// toggle the value
	joined = !joined;
	// set the new value
	gtk_tree_store_set (GTK_TREE_STORE (model), &iter, JOINED_COLUMN, joined, -1);

	// handle the joining/closing of the selected document
	pDlg->eventJoin(
			static_cast<Buddy*>(buddy), 
			static_cast<DocHandle*>(doc_handle), joined
		);

	// clean up
	gtk_tree_path_free (path);
}

XAP_Dialog * AP_UnixDialog_CollaborationJoin::static_constructor(XAP_DialogFactory * pFactory, XAP_Dialog_Id id)
{
	return static_cast<XAP_Dialog *>(new AP_UnixDialog_CollaborationJoin(pFactory, id));
}
pt2Constructor ap_Dialog_CollaborationJoin_Constructor = &AP_UnixDialog_CollaborationJoin::static_constructor;

AP_UnixDialog_CollaborationJoin::AP_UnixDialog_CollaborationJoin(XAP_DialogFactory * pDlgFactory, XAP_Dialog_Id id)
	: AP_Dialog_CollaborationJoin(pDlgFactory, id),
	m_wWindowMain(NULL),
	m_wAddBuddy(NULL),
	m_wDeleteBuddy(NULL),
	m_wModel(NULL),
	m_wBuddyTree(NULL)
{
}

void AP_UnixDialog_CollaborationJoin::runModal(XAP_Frame * pFrame)
{
	UT_return_if_fail(pFrame);
	
    // Build the dialog's window
	m_wWindowMain = _constructWindow();
	UT_return_if_fail(m_wWindowMain);

	_populateWindowData();

	// refresh the contents, just to be up to date
	eventRefresh();
	
	switch ( abiRunModalDialog ( GTK_DIALOG(m_wWindowMain),
								 pFrame, this, GTK_RESPONSE_CLOSE, false ) )
	{
		/*case GTK_RESPONSE_CLOSE:
			m_answer = AP_UnixDialog_CollaborationAccounts::a_CLOSE;
			break;
		default:
			m_answer = AP_UnixDialog_CollaborationAccounts::a_CLOSE;
			break;*/
	}

	abiDestroyWidget(m_wWindowMain);
}

/*****************************************************************/
GtkWidget * AP_UnixDialog_CollaborationJoin::_constructWindow(void)
{
	GtkWidget* window;
	const XAP_StringSet * pSS = XAP_App::getApp()->getStringSet();
	
	// get the path where our glade file is located
	XAP_UnixApp * pApp = static_cast<XAP_UnixApp*>(XAP_App::getApp());
	UT_String glade_path( pApp->getAbiSuiteAppGladeDir() );
	glade_path += "/ap_UnixDialog_CollaborationJoin.glade";
	// load the dialog from the glade file
	GladeXML *xml = abiDialogNewFromXML( glade_path.c_str() );
	if (!xml)
		return NULL;
	
	// Update our member variables with the important widgets that 
	// might need to be queried or altered later
	window = glade_xml_get_widget(xml, "ap_UnixDialog_CollaborationJoin");
	m_wAddBuddy = glade_xml_get_widget(xml, "btAddBuddy");
	m_wDeleteBuddy = glade_xml_get_widget(xml, "btDeleteBuddy");
	m_wRefresh = glade_xml_get_widget(xml, "btRefresh");	
	m_wBuddyTree = glade_xml_get_widget(xml, "tvBuddies");

	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	gtk_widget_set_sensitive(m_wAddBuddy, pManager->getAccounts().getItemCount() != 0);
	gtk_widget_set_sensitive(m_wDeleteBuddy, false); // TODO: implement this
	gtk_widget_set_sensitive(m_wRefresh, true);	

	// set the dialog title
	// TODO
	
	// localize the strings in our dialog, and set tags for some widgets
	// TODO

	// connect our signals
	g_signal_connect(G_OBJECT(m_wAddBuddy),
							"clicked",
							G_CALLBACK(s_add_buddy_clicked),
							static_cast<gpointer>(this));

	g_signal_connect(G_OBJECT(m_wRefresh),
							"clicked",
							G_CALLBACK(s_refresh_clicked),
							static_cast<gpointer>(this));

	return window;
}

void AP_UnixDialog_CollaborationJoin::_populateWindowData()
{
	GtkTreeSelection *sel;
	_setModel(_constructModel());
	
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (m_wBuddyTree), true);

	// get the current selection
	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (m_wBuddyTree));
	gtk_tree_selection_set_mode (sel, GTK_SELECTION_BROWSE);
	//gtk_tree_selection_set_select_function (sel, tree_select_filter, NULL, NULL);
	
	
	gint col_offset;
	GtkTreeViewColumn* column;
	GtkCellRenderer *renderer;
	renderer = gtk_cell_renderer_text_new ();
	col_offset = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (m_wBuddyTree), 
												-1,	"Buddy", 
												renderer, 
												"text", DESCRIPTION_COLUMN,
												(void*)NULL);
	column = gtk_tree_view_get_column (GTK_TREE_VIEW (m_wBuddyTree), col_offset - 1);	
	
	renderer = gtk_cell_renderer_toggle_new ();
	g_object_set (renderer, "xalign", 0.0, NULL);
	g_signal_connect (renderer, "toggled", G_CALLBACK (joined_toggled), this);
	col_offset = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (m_wBuddyTree), 
												-1,	"Joined", 
												renderer, 
												"active", JOINED_COLUMN,
												"visible", VISIBLE_COLUMN,
												(void*)NULL);
	column = gtk_tree_view_get_column (GTK_TREE_VIEW (m_wBuddyTree), col_offset - 1);												
	gtk_tree_view_column_set_fixed_width (GTK_TREE_VIEW_COLUMN (column), 50);
	gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);

	gtk_tree_view_expand_all (GTK_TREE_VIEW (m_wBuddyTree));
	gtk_widget_show_all(m_wBuddyTree);
}

GtkTreeStore* AP_UnixDialog_CollaborationJoin::_constructModel()
{
	GtkTreeIter iter;
	GtkTreeStore* model = gtk_tree_store_new (NUM_COLUMNS, G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_POINTER, G_TYPE_POINTER, G_TYPE_BOOLEAN);

	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	const UT_GenericVector<AccountHandler*>& accounts = pManager->getAccounts();

	for (UT_sint32 i = 0; i < accounts.getItemCount(); i++)
	{
		UT_DEBUGMSG(("Getting buddies for account: %s of type %s\n", 
				accounts.getNthItem(i)->getDescription().utf8_str(), 
				accounts.getNthItem(i)->getDisplayType().utf8_str()
			));
			
		// add all buddies belonging to this account
		for (UT_sint32 j = 0; j < accounts.getNthItem(i)->getBuddies().size(); j++)
		{
			const Buddy* pBuddy = accounts.getNthItem(i)->getBuddies()[j];
		
			gtk_tree_store_append (model, &iter, NULL);
			gtk_tree_store_set (model, &iter, 
					DESCRIPTION_COLUMN, pBuddy->getDescription().utf8_str(), 
					JOINED_COLUMN, false, /* joined */
					DOCHANDLE_COLUMN, 0, /* dochandle */
					BUDDY_COLUMN, 0, /* buddy */
					VISIBLE_COLUMN, false, /* visibility */
					-1);
					
			// add all documents for this buddy
			GtkTreeIter child_iter;
			for (const DocTreeItem* item = pBuddy->getDocTreeItems(); item; item = item->m_next)
			{
				if (item->m_docHandle)
				{
					UT_DEBUGMSG(("DocHandle document name: %s\n", item->m_docHandle->getName().utf8_str()));
				
					// TODO: handle the DocTreeItem type
					gtk_tree_store_append (model, &child_iter, &iter);
					gtk_tree_store_set (model, &child_iter, 
							DESCRIPTION_COLUMN, (item->m_docHandle ? item->m_docHandle->getName().utf8_str() : "null"),
							JOINED_COLUMN, pManager->isActive(item->m_docHandle->getSessionId()),
							DOCHANDLE_COLUMN, item->m_docHandle,
							BUDDY_COLUMN, pBuddy,
							VISIBLE_COLUMN, true,
							-1);
				}
				else
					UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
			}
		}
	}
	return model;
}

void AP_UnixDialog_CollaborationJoin::_setModel(GtkTreeStore* model)
{
	// TODO: free the old model
	m_wModel = model;
	gtk_tree_view_set_model(GTK_TREE_VIEW (m_wBuddyTree), GTK_TREE_MODEL(m_wModel));
	gtk_tree_view_expand_all (GTK_TREE_VIEW (m_wBuddyTree));
	gtk_widget_show_all(m_wBuddyTree);	
}

void AP_UnixDialog_CollaborationJoin::eventAddBuddy()
{
	_eventAddBuddy();
	
	// Update the dialog
	// We should only refresh a view, not reload then entire contents
	_setModel(_constructModel());
}

void AP_UnixDialog_CollaborationJoin::eventRefresh()
{
	// TODO: we really should refresh the buddies here as well, 
	// as they could pop up automatically as well (for example with a 
	// avahi backend)
	_refreshAllDocHandlesAsync();
}

void AP_UnixDialog_CollaborationJoin::eventJoin(Buddy* pBuddy, DocHandle* pDocHandle, bool joined)
{
	UT_DEBUGMSG(("AP_UnixDialog_CollaborationJoin::eventJoin()\n"));
	UT_return_if_fail(pBuddy);
	UT_return_if_fail(pDocHandle);

	if (joined)
		_join(pBuddy, pDocHandle);
	else
		_disjoin(pBuddy, pDocHandle);
}

void AP_UnixDialog_CollaborationJoin::signal(const Event& event, const Buddy* pSource)
{
	UT_DEBUGMSG(("AP_UnixDialog_CollaborationJoin::signal()\n"));
	switch (event.getType())
	{
		case Event::AccountNew:
		case Event::AccountAddBuddy:
		case Event::AccountDeleteBuddy:
		case Event::AccountBuddyOnline:
		case Event::AccountBuddyOffline:
			// FIXME: ick ick ick! (I shouldn't need to explain this)
			_setModel(_constructModel());
			break;
		case Event::AccountBuddyAddDocument:
			{
				// FIXME: until the event actually tells us which session
				// was started or closed, we'll simply request the whole 
				// session list from the remote buddy
				// FIXME: we should be able to refresh the list for a specific buddy/session
				const AccountBuddyAddDocumentEvent* pAde = static_cast<const AccountBuddyAddDocumentEvent*>(&event);
				DocHandle* pDocHandler = pAde->getDocHandle();
				// TODO: only refresh the buddy belonging to this handle
				_setModel(_constructModel());
			}
			break;	
		case Event::StartSession:
			// FIXME: until the event actually tells us which session
			// was started, we'll simply request the whole 
			// session list from the remote buddy
			// FIXME: we should be able to refresh the list for a specific buddy/session
			_refreshAllDocHandlesAsync();
			break;
		case Event::CloseSession:
			// TODO: only remove the closed session from the list
			_setModel(_constructModel());
			break;
		default:
			// we will ignore the rest
			break;
	}
}
