/* 
 * Copyright (C) 2005 by Martin Sevior
 * Copyright (C) 2006 by Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "pd_Document.h"
#include "ut_assert.h"
#include "ut_debugmsg.h"
#include "xap_Frame.h"
#include "xap_Strings.h"
#include "xap_App.h"
#include "ut_path.h"
#include "xap_Dlg_MessageBox.h"

#include "XMPPAccountHandler.h"
#include "XMPPBuddy.h"

#include <handlers/xp/AccountEvent.h>
#include <handlers/xp/SessionEvent.h>

#include <xp/AbiCollabSessionManager.h>
#include <xp/AbiCollab.h>
#include <xp/AbiCollab_Plugin.h>

#include <xp/AbiCollab_Packet.h>
#include <xp/EventPacket.h>

static LmHandlerResult presence_handler(LmMessageHandler      *handler,
					LmConnection          *connection,
					LmMessage             *m,
					gpointer               user_data)
{
	XMPPAccountHandler * pHandler = static_cast<XMPPAccountHandler *>(user_data);
	LmMessageNode* node = lm_message_get_node(m);
	if (node)
	{
		const char* from = static_cast<const char*>(lm_message_node_get_attribute (node, "from"));
		if (from)
		{
			const gchar* type = lm_message_node_get_attribute (node, "type");
			
			if (type && strcmp(type, "unavailable") == 0)
			{
				UT_DEBUGMSG(("Disconnect presence from %s\n", from));
				// TODO: handle this
			}
			else
			{
				UT_DEBUGMSG(("Connect presence from %s\n", from));
				// TODO: handle this
			}
		}
		else
			UT_DEBUGMSG(("presence message without from\n"));
	}
	return LM_HANDLER_RESULT_REMOVE_MESSAGE;
}

static LmHandlerResult stream_error_handler(LmMessageHandler      *handler,
					LmConnection          *connection,
					LmMessage             *m,
					gpointer               user_data)
{
	XMPPAccountHandler * pHandler = static_cast<XMPPAccountHandler *>(user_data);
	UT_return_val_if_fail(pHandler, LM_HANDLER_RESULT_REMOVE_MESSAGE);	
	
	LmMessageNode* node = lm_message_get_node(m);
	UT_DEBUGMSG(("Stream error message |%s|=|%s|\n", node->name, node->value));
	
	// FIXME: for now, we assume we are disconnected; not sure if this is always the case
	pHandler->disconnect();
	
	return LM_HANDLER_RESULT_REMOVE_MESSAGE;

}

static LmHandlerResult chat_handler(LmMessageHandler      *handler,
				    LmConnection          *connection,
				    LmMessage             *m,
				    gpointer               user_data)
{
	XMPPAccountHandler * pHandler = static_cast<XMPPAccountHandler *>(user_data);
	UT_return_val_if_fail(pHandler, LM_HANDLER_RESULT_REMOVE_MESSAGE);
	
	/* TODO: we should run run through all the nodes to find the message node */
	LmMessageNode* node = lm_message_get_node(m);
	if (strcmp(node->name, "message") == 0)
	{
		for (LmMessageNode* child = node->children; child != 0; child = child->next)
		{
			if (strcmp(child->name, "body") == 0)
			{
				/*
				   Note: we don't trust a message body with a from-address in it, as it could be forged. 
				   Instead get the from-address directly from the LmMessage
				*/
				std::string buddy = lm_message_node_get_attribute (m->node, "from");
				std::string::size_type pos  = buddy.find_last_of("/");
				if (pos != std::string::npos)
					buddy.resize(pos);

				// TODO: check the resource as an additional sanity check

				UT_DEBUGMSG(("chat_handler(): Got a message from buddy: %s\n", buddy.c_str()));
				RawPacket* pRp = new RawPacket();
				pRp->buddy = new XMPPBuddy(pHandler, buddy.c_str());
				pRp->packet = child->value;
				pHandler->handleMessage(pRp);
				break;
			}
		}
	}
	
	return LM_HANDLER_RESULT_REMOVE_MESSAGE;
}

/*
 * XMPPAccountHandler
 */

XMPPAccountHandler::XMPPAccountHandler():
	m_pConnection(NULL),
	m_pPresenceHandler(NULL),
	m_pStreamErrorHandler(NULL),
	m_pChatHandler(NULL),
	m_bLoggedIn(false)
{
}

XMPPAccountHandler::~XMPPAccountHandler()
{
	disconnect();
}

UT_UTF8String XMPPAccountHandler::getDescription()
{
	const std::string username = getProperty("username");
	const std::string server = getProperty("server");	
	return UT_UTF8String_sprintf("%s@%s", username.c_str(), server.c_str());
}

UT_UTF8String XMPPAccountHandler::getDisplayType()
{
	return "Jabber (XMPP)";
}

UT_UTF8String XMPPAccountHandler::getStorageType()
{
	return "com.abisource.abiword.abicollab.handler.xmpp";
}

UT_sint32 XMPPAccountHandler::connect()
{
	if (m_bLoggedIn)
		return 1; // already connected
	
	UT_return_val_if_fail(m_pConnection == 0, -1); // internal error, we still seem to be connected

	XAP_Frame *pFrame = XAP_App::getApp()->getLastFocussedFrame();
	UT_return_val_if_fail(pFrame, -1);	
	
	const std::string server = getProperty("server");
	const std::string port = getProperty("port"); // TODO: unused atm
	const std::string username = getProperty("username");
	const std::string password = getProperty("password");
	const std::string resource = getProperty("resource");
	
	UT_DEBUGMSG(("Connecting to server: |%s|, resource: |%s|\n", server.c_str(), resource.c_str()));
	m_pConnection = lm_connection_new(server.c_str());
	if (m_pConnection)
	{
		GError* error = NULL;

		// TODO: make this non-blocking
		if (!lm_connection_open_and_block(m_pConnection, &error)) 
		{
			UT_DEBUGMSG(("Failed to open: %s\n", (error ? error->message : "") ));
			lm_connection_unref(m_pConnection);
			m_pConnection = NULL;
			
			// inform the user of the connection failure
			// TODO: this shouldn't be here, the caller should handle this
			UT_UTF8String msg;
			// TODO: make this localizable
			UT_UTF8String_sprintf(msg, "Error while connecting to %s: %s\n", server.c_str(), (error ? error->message : "")); 
			pFrame->showMessageBox(msg.utf8_str(), XAP_Dialog_MessageBox::b_O, XAP_Dialog_MessageBox::a_OK);
						
			return 2; // no connection
		}
		UT_DEBUGMSG(("connect() - connection established!\n"));
	
		// TODO: make this non-blocking
		if (!lm_connection_authenticate_and_block (m_pConnection, username.c_str(), password.c_str(),
					resource.c_str(), &error))
		{
			UT_DEBUGMSG(("connect() - couldn't authenticate with '%s' '%s':\n%s\n", 
					username.c_str(), password.c_str(), (error ? error->message : "")));
					
			lm_connection_close(m_pConnection, NULL);
			lm_connection_unref(m_pConnection);
			m_pConnection = NULL;
			
			// inform the user of the authentication failure
			// TODO: this shouldn't be here, the caller should handle this
			UT_UTF8String msg;
			// TODO: make this localizable
			UT_UTF8String_sprintf(msg, "Error while connecting to %s: %s\n", server.c_str(), (error ? error->message : "")); 
			pFrame->showMessageBox(msg.utf8_str(), XAP_Dialog_MessageBox::b_O, XAP_Dialog_MessageBox::a_OK);

			return 3; // Invalid authentication
		}
		UT_DEBUGMSG(("connect() - user (%s) authenticated!\n", username.c_str()));
	
		// Register message handler for presence messages
		m_pPresenceHandler = lm_message_handler_new((LmHandleMessageFunction)presence_handler, reinterpret_cast< gpointer >(this), NULL);
		lm_connection_register_message_handler(m_pConnection, m_pPresenceHandler, LM_MESSAGE_TYPE_PRESENCE, LM_HANDLER_PRIORITY_NORMAL);
	
		// Register message handler for stream error messages
		m_pStreamErrorHandler = lm_message_handler_new((LmHandleMessageFunction)stream_error_handler, reinterpret_cast< gpointer >(this), NULL);
		lm_connection_register_message_handler(m_pConnection, m_pStreamErrorHandler, LM_MESSAGE_TYPE_STREAM_ERROR, LM_HANDLER_PRIORITY_NORMAL);
	
		// Register message handler for chat messages
		m_pChatHandler = lm_message_handler_new((LmHandleMessageFunction)chat_handler, reinterpret_cast< gpointer >(this), NULL);
		lm_connection_register_message_handler(m_pConnection, m_pChatHandler, LM_MESSAGE_TYPE_MESSAGE, LM_HANDLER_PRIORITY_NORMAL);
	
		// Send presence message to server
		LmMessage* m = lm_message_new_with_sub_type(NULL, LM_MESSAGE_TYPE_PRESENCE, LM_MESSAGE_SUB_TYPE_NOT_SET);
		if (!lm_connection_send(m_pConnection, m, &error)) 
		{
			UT_DEBUGMSG(("Presence message could not be sent: %s", (error ? error->message : "") ));
			lm_connection_close(m_pConnection, NULL);
			lm_connection_unref(m_pConnection);
			m_pConnection = NULL;
			
			// FIXME: unregister the message handlers!
			// ...
			
			// inform the user of the sending error
			// TODO: this shouldn't be here, the caller should handle this
			UT_UTF8String msg;
			// TODO: make this localizable
			UT_UTF8String_sprintf(msg, "Error while connecting to %s: %s\n", server.c_str(), (error ? error->message : "")); 
			pFrame->showMessageBox(msg.utf8_str(), XAP_Dialog_MessageBox::b_O, XAP_Dialog_MessageBox::a_OK);			
			
			return 4; // error sending msg
		}
		lm_message_unref(m);

		m_bLoggedIn = true;

		// we are connected now, time to start and send out messages (such as events) anymore
		AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
		if (pManager)
		{
			pManager->registerEventListener(this);
		}
		else
			UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
	
		// signal all listeners we are logged in
		AccountOnlineEvent event;
		// TODO: fill the event
		AbiCollabSessionManager::getManager()->signal(event);
	}
	else
	{
		UT_DEBUGMSG(("lm_connection_new error for server: %s", server.c_str()));
		
		// inform the user of the sending error
		// TODO: this shouldn't be here, the caller should handle this
		UT_UTF8String msg;
		// TODO: make this localizable
		UT_UTF8String_sprintf(msg, "Error while connecting to %s: internal error\n", server.c_str()); 
		pFrame->showMessageBox(msg.utf8_str(), XAP_Dialog_MessageBox::b_O, XAP_Dialog_MessageBox::a_OK);				
		
		return -1; // internal error
	}
		
	return 0;
}

bool XMPPAccountHandler::disconnect()
{
	UT_DEBUGMSG(("XMPPAccountHandler::disconnect()\n"));

	// unregister and destroy the message handler callbacks
	if (m_pPresenceHandler)
	{
		lm_connection_unregister_message_handler(m_pConnection, m_pPresenceHandler, LM_MESSAGE_TYPE_PRESENCE);
		lm_message_handler_unref(m_pPresenceHandler);
		m_pPresenceHandler = NULL;
	}
	
	if (m_pStreamErrorHandler)
	{
		lm_connection_unregister_message_handler(m_pConnection, m_pStreamErrorHandler, LM_MESSAGE_TYPE_STREAM_ERROR);
		lm_message_handler_unref(m_pStreamErrorHandler);
		m_pStreamErrorHandler = NULL;
	}

	if (m_pChatHandler)
	{
		lm_connection_unregister_message_handler(m_pConnection, m_pChatHandler, LM_MESSAGE_TYPE_MESSAGE);
		lm_message_handler_unref(m_pChatHandler);
		m_pChatHandler = NULL;
	}
	
	lm_connection_close(m_pConnection, NULL);
	lm_connection_unref(m_pConnection);
	m_pConnection = NULL;
	m_bLoggedIn = false;
	
	// signal all listeners we are logged out
	AccountOfflineEvent event;
	// TODO: fill the event
	AbiCollabSessionManager::getManager()->signal(event);


	// we are disconnected now, no need to sent out messages (such as events) anymore
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	if (pManager)
	{	
		pManager->unregisterEventListener(this);	
	}
	else
		UT_ASSERT(UT_SHOULD_NOT_HAPPEN);	
	
	
	return true;
}

bool XMPPAccountHandler::autoConnect()
{
	const std::string resource = getProperty("autoconnect");
	return strcmp(resource.c_str(), "true") == 0;
}

bool XMPPAccountHandler::_send(const UT_UTF8String& packet)
{
	GError* error = NULL;
	const std::string resource = getProperty("resource");
		
	for (UT_sint32 i = 0; i < getBuddies().getItemCount(); i++)
	{
		const XMPPBuddy* pBuddy = reinterpret_cast<const XMPPBuddy*>(getBuddies().getNthItem(i));
		if (pBuddy)
		{
			if (!_send(packet, *pBuddy))
			{
				UT_DEBUGMSG(("Error while sending message to '%s':\n%s\n",
						pBuddy->getName().utf8_str(), (error ? error->message : "") ));
			}
		}
	}

	return true;
}

bool XMPPAccountHandler::_send(const UT_UTF8String& packet, const Buddy& buddy)
{
	if (!m_pConnection)
		return false;

	GError* error = NULL;
	
	// TODO: make sure these properties are always there
	const std::string resource = getProperty("resource");
	const std::string server = getProperty("server");

	// fully qualified address
	UT_UTF8String fqa = buddy.getName();
	fqa += "/";
	fqa += resource.c_str();
	
	UT_DEBUGMSG(("Sending packet |%s| to |%s|\n", packet.utf8_str(), fqa.utf8_str()));
	LmMessage* m = lm_message_new (fqa.utf8_str(), LM_MESSAGE_TYPE_MESSAGE);
	lm_message_node_add_child (m->node, "body", packet.utf8_str());
	if (!lm_connection_send (m_pConnection, m, &error)) {
		UT_DEBUGMSG(("Error while sending message to '%s':\n%s\n",
				packet.utf8_str(), (error ? error->message : "") ));
		return false;
	}
	lm_message_unref(m);

	return true;
}

void XMPPAccountHandler::handleMessage(RawPacket* pRp)
{
	UT_return_if_fail(pRp);
	UT_return_if_fail(pRp->buddy);
			
	// TODO: most likely, this packet parsing code can be moved to AccountHandler itself

	// First capture any request that is destined for the account handler itself
	xmlDocPtr doc = xmlReadMemory (pRp->packet.utf8_str(), pRp->packet.length(), 0, "UTF-8", 0);
	UT_return_if_fail(doc);
	
	xmlNode* protocolNode = xmlDocGetRootElement(doc);
	UT_return_if_fail(protocolNode);
	
	// we only expect 1 root element - do we really? - MARCM
	UT_ASSERT(protocolNode->next == 0);
		
	if (strcmp(reinterpret_cast<const char*>(protocolNode->name), "abicollab") == 0) 
	{
		UT_UTF8String sPacketProtocolVersion = reinterpret_cast<char *>(xmlGetProp(protocolNode, reinterpret_cast<const xmlChar*>("protocol")));
		if (sPacketProtocolVersion == ABICOLLAB_PROTOCOL_VERSION)
		{
			xmlNode* packetTypeNode = protocolNode->children;
			if (packetTypeNode)
			{
				if (strcmp(reinterpret_cast<const char*>(packetTypeNode->name), "session") == 0)
				{
					// this is a session packet
					AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
					pManager->processSessionPacket(*this, packetTypeNode, pRp->buddy->getName());
				}
				else if (strcmp(reinterpret_cast<const char*>(packetTypeNode->name), "event") == 0)
				{
					// this is an event packet
					AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
					pManager->processEventPacket(*this, packetTypeNode, pRp->buddy->getName());
				}
				else if (strcmp(reinterpret_cast<const char*>(packetTypeNode->name), "handler") == 0)
				{
					// yay, this is a packet for us!
					xmlNode* handlerNode = packetTypeNode->children;
					if (handlerNode)
					{
						if(strcmp(reinterpret_cast<const char*>(handlerNode->name), "GetSessionsRequest") == 0)
						{
							_handleGetSessionsRequest(handlerNode, pRp);
						}
						else if (strcmp(reinterpret_cast<const char*>(handlerNode->name), "GetSessionsResponse") == 0)
						{
							_handleGetSessionsResponse(handlerNode, pRp);
						}
						else if(strcmp(reinterpret_cast<const char*>(handlerNode->name), "JoinSessionRequest") == 0)
						{
							_handleJoinSessionRequest(handlerNode, pRp);
						}
						else if(strcmp(reinterpret_cast<const char*>(handlerNode->name), "JoinSessionResponse") == 0)
						{
							_handleJoinSessionResponse(handlerNode, pRp);
						}
					}
				}
				else
				{
					// huh?!
					UT_ASSERT(false);
				}
			}
		}
		else
		{
			// TODO: protocol version mismatch!
		}
	}

	DELETEP(pRp);
	xmlFreeDoc(doc);
}

Buddy* XMPPAccountHandler::constructBuddy(const PropertyMap& vProps)
{
	PropertyMap::const_iterator pos = vProps.find("name");
	if (pos != vProps.end())
	{
		UT_return_val_if_fail(pos->second.size() > 0, 0);
		UT_DEBUGMSG(("Constructing buddy (%s)\n", pos->second.c_str()));
		return new XMPPBuddy(this, pos->second.c_str());
	}
	UT_ASSERT_HARMLESS(UT_NOT_REACHED);
	return 0;
}

void XMPPAccountHandler::getSessionsAsync(const Buddy& buddy)
{
	UT_UTF8String sDocRequest = "<GetSessionsRequest/>";
	send(NULL, sDocRequest, PT_Handler, buddy);
}

void XMPPAccountHandler::joinSessionAsync(const Buddy& buddy, DocHandle& docHandle)
{
	UT_UTF8String sDocRequest = "<JoinSessionRequest>";
	sDocRequest += "<sessionId>";
	sDocRequest += docHandle.getSessionId();
	sDocRequest += "</sessionId>";
	sDocRequest += "</JoinSessionRequest>";
	send(NULL, sDocRequest, PT_Handler, buddy);
}

void XMPPAccountHandler::signal(const Event& event, const Buddy* pSource)
{
	UT_DEBUGMSG(("XMPPAccountHandler::signal()\n"));

	// broadcast this event over our network (if applicable for each message type)
	EventPacket* pPacket = event.constructPacket();
	if (pPacket)
	{
		UT_GenericVector<Buddy*> vRecipients = 
			(event.isBroadcast() ? getBuddies() : event.getRecipients());
		
		for (UT_sint32 i = 0; i < vRecipients.getItemCount(); i++)
		{
			Buddy* pRecipient = vRecipients.getNthItem(i);
			if (pRecipient)
			{
				if (!pSource || 
					(pSource && pRecipient->getName() != pSource->getName())
					)
				{
					send(0, pPacket->serialize(), pPacket->getType(), *pRecipient);
				}
				else
				{
					// the event came originated at this buddy, so make sure not to send it
					// back to him, as it would result in a broadcast storm and
					// kill the network really fast
				}
			}
		}
	}
}

/* 
 * packet handling functions 
 */

void XMPPAccountHandler::_handleGetSessionsRequest(xmlNode* parent, RawPacket* pRp)
{
	UT_DEBUGMSG(("XMPPAccountHandler::_handleGetSessionsRequest()\n"));
	UT_return_if_fail(pRp);

	// A remote user has requested our open and connected documents
	// Return an xml-ized list of them.
	UT_UTF8String sDocResponse = "<GetSessionsResponse>";
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	const UT_GenericVector<AbiCollab *> sessions = pManager->getSessions();
	for (UT_sint32 i = 0; i < sessions.getItemCount(); i++)
	{
		AbiCollab* pSession = sessions.getNthItem(i);
		if (pSession && pSession->isLocallyControlled())
		{
			const PD_Document * pDoc = pSession->getDocument();
			if (pDoc)
			{
				sDocResponse += "<session name=\"";
				// Let's use the document name for the session name
				// TODO: check the metadata first for a nicer document name
				if (pDoc->getFilename())
				{
					UT_UTF8String sDocumentBaseName(UT_go_basename_from_uri(pDoc->getFilename()));
					sDocResponse += sDocumentBaseName;
				}			
				sDocResponse += "\" id=\"";
				sDocResponse += pSession->getSessionId();
				sDocResponse += "\"/>";
			}
			else
				UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
		}
	}
	sDocResponse +="</GetSessionsResponse>";
	send(NULL, sDocResponse, PT_Handler, *pRp->buddy);
}

void XMPPAccountHandler::_handleGetSessionsResponse(xmlNode* parent, RawPacket* pRp)
{
	UT_DEBUGMSG(("XMPPAccountHandler::_handleGetSessionsResponse()\n"));
	UT_return_if_fail(parent);
	UT_return_if_fail(pRp);
	
	AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
	UT_return_if_fail(pManager);
	
	Buddy* pBuddy = getBuddy(pRp->buddy->getName());
	if (pBuddy != NULL)
	{
		UT_GenericVector<DocHandle*> vDocHandles;
		for (xmlNode* document_node = parent->children; document_node; document_node = document_node->next)
		{
			if (document_node->type == XML_ELEMENT_NODE)
			{
				if (strcmp(reinterpret_cast<const char*>(document_node->name), "session") == 0)
				{
					UT_UTF8String sSessionId = reinterpret_cast<char *>(xmlGetProp(document_node, reinterpret_cast<const xmlChar*>("id")));
					UT_DEBUGMSG(("Got a session with id %s\n", sSessionId.utf8_str()));
					UT_UTF8String sSessionName = reinterpret_cast<char *>(xmlGetProp(document_node, reinterpret_cast<const xmlChar*>("name"))); 
					
					DocHandle* pDocHandle = new DocHandle(sSessionId, sSessionName);
					vDocHandles.addItem(pDocHandle);
				}
			}
		}
		pManager->setDocumentHandles(*pBuddy, vDocHandles);
	}
}

void XMPPAccountHandler::_handleJoinSessionRequest(xmlNode* parent, RawPacket* pRp)
{
	UT_DEBUGMSG(("XMPPAccountHandler::_handleJoinSessionRequest()\n"));
	UT_return_if_fail(pRp);
	
	UT_UTF8String sSessionId;
	
	// parse all document properties in this request
	for (xmlNode* prop = parent->children; prop; prop = prop->next)
	{
		if (prop->type == XML_ELEMENT_NODE)
		{
			if (strcmp(reinterpret_cast<const char*>(prop->name), "sessionId") == 0)
			{
				sSessionId = reinterpret_cast<const char*>(xmlNodeGetContent(prop));
				UT_DEBUGMSG(("Incoming JoinSessionRequest for session id |%s|\n", sSessionId.utf8_str()));
			}
		}
	}
	
	if (sSessionId != "")
	{
		// TODO: ask the user to authorize this request
		bool bAuthorized = true;
			
		if (bAuthorized)
		{
			AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
			AbiCollab* pAbiCollab = pManager->getSessionFromSessionId(sSessionId);
			if (pAbiCollab)
			{
				PD_Document* pDoc = pAbiCollab->getDocument();
			
				UT_UTF8String sBase64GzDocument;
				if (AbiCollabSessionManager::serializeDocument(pDoc, sBase64GzDocument) == UT_OK)
				{
					UT_UTF8String sDocResponse = "<JoinSessionResponse>\n";
					sDocResponse += "<document id=\"";
					sDocResponse += pDoc->getDocUUIDString();
					sDocResponse += "\" sessionId=\"";
					sDocResponse += sSessionId.utf8_str();
					sDocResponse += "\" name=\"";
					// TODO: check the metadata first for a nicer document name
					if (pDoc->getFilename())
					{
						UT_UTF8String sDocumentBaseName(UT_go_basename_from_uri(pDoc->getFilename()));
						sDocResponse += sDocumentBaseName;
					}
					sDocResponse +=	"\">";
					sDocResponse += sBase64GzDocument;
					sDocResponse += "</document>\n";
					sDocResponse +="</JoinSessionResponse>";
					send(NULL, sDocResponse, PT_Handler, *pRp->buddy);
				}
				
				// NOTE: this is kinda wacky atm, but for now, when someone has requested a document,
				// we'll consider him joined to the session as well
				Buddy* pCollaborator = pRp->buddy; // ICK ICK, please please implement a proper clone() function
				
				if (pCollaborator)
				{
					// check if we already know this buddy
					Buddy* existing = getBuddy(pCollaborator->getName());
					if (!existing)
					{
						// we don't know this buddy yet; add this one as a volatile buddy
						pCollaborator->setVolatile(true);
						addBuddy(pCollaborator);
					}
				
					// add this buddy to the collaboration session
					pAbiCollab->addCollaborator(pCollaborator);
					pRp->buddy = 0; // ICK ICK, please please implement a proper clone() function
				}
				else
					UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
			}
			else
				UT_DEBUGMSG(("Requesting a session with an unknown id (%s)!\n", sSessionId.utf8_str()));
		}
	}
	else
		UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
}

void XMPPAccountHandler::_handleJoinSessionResponse(xmlNode* parent, RawPacket* pRp)
{
	UT_DEBUGMSG(("XMPPAccountHandler::_handleJoinSessionResponse()\n"));
	UT_return_if_fail(pRp);

	// we only expect 1 document per response
	UT_ASSERT(parent->next == 0);
	
	// get the document's contents
	UT_UTF8String sSessionId;
	UT_UTF8String sDocumentName;
	UT_UTF8String sDocumentContent;
	for (xmlNode* prop = parent->children; prop; prop = prop->next)	
	{
		if (prop->type == XML_ELEMENT_NODE)
		{
			if (strcmp(reinterpret_cast<const char*>(prop->name), "document") == 0)
			{
				// get the session id
				char* sz = (char *)xmlGetProp(prop, (const xmlChar *)"sessionId");
				sSessionId = sz;
				FREEP(sz);
				
				// get the session id
				sz = (char *)xmlGetProp(prop, (const xmlChar *)"name");
				sDocumentName = sz;
				FREEP(sz);
				
				// get the document id
				// TODO: do we need this? if so, then implement this
				
				// get the document contents
				sDocumentContent = reinterpret_cast<const char*>(xmlNodeGetContent(prop));
			}
		}
	}
	
	Buddy* pCollaborator = pRp->buddy;			
	if (pCollaborator)
	{
		AbiCollabSessionManager* pManager = AbiCollabSessionManager::getManager();
		if (pManager)
		{
			PD_Document* pDoc = 0;
			if (AbiCollabSessionManager::deserializeDocument(&pDoc, sDocumentContent) == UT_OK)
			{
				if (pDoc)
				{
					// NOTE: we could adopt the same document name here, but i'd
					// rather not at the moment - MARCM
					pDoc->forceDirty();
					pManager->joinSession(sSessionId, pDoc, pCollaborator);
				}
				else
					UT_DEBUGMSG(("XMPPAccountHandler::_handleJoinSessionResponse() - deserializing document failed!\n"));
			}
		}
	}
}

