/* AbiSource
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */
 
// Class definition include
#include "OD_Office_Styles.h"

// Internal inlcudes
#include "OD_Style_Style.h"
#include "OD_Style_PageLayout.h"
#include "OD_Style_MasterPage.h"
#include "OD_Style_List.h"
#include "OD_ListLevelStyle.h"
#include "OD_ElementStack.h"
 
// AbiWord includes
#include <ut_misc.h>
#include <pd_Document.h>
#include <ut_debugmsg.h>





/**
 * Destructor
 */
OD_Office_Styles::~OD_Office_Styles() {
    
    UT_GenericVector<OD_Style_Style*>* pStyleVector;
    UT_GenericVector<OD_Style_PageLayout*>* pPageLayoutVector;
    UT_GenericVector<OD_Style_MasterPage*>* pMasterPageVector;
    UT_GenericVector<OD_Style_List*>* pListStyleVector;
    UT_uint32 i, count;
    
    
    pStyleVector = m_textStyleStyles.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }
    

    pStyleVector = m_paragraphStyleStyles.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }


    pStyleVector = m_sectionStyleStyles.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }
    
    
    pStyleVector = m_graphicStyleStyles.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }
    
    
    
    
    pStyleVector = m_textStyleStyles_contentStream.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }
    

    pStyleVector = m_paragraphStyleStyles_contentStream.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }


    pStyleVector = m_sectionStyleStyles_contentStream.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }
    
    
    pStyleVector = m_graphicStyleStyles_contentStream.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }
    
    
    
    
    
    
    pListStyleVector = m_listStyles.enumerate();
    count = pListStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pListStyleVector)[i];
    }
    
    
    pPageLayoutVector = m_pageLayoutStyles.enumerate();
    count = pPageLayoutVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pPageLayoutVector)[i];
    }
    
    
    pMasterPageVector = m_masterPageStyles.enumerate();
    count = pMasterPageVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pMasterPageVector)[i];
    }

    DELETEP(m_pParagraphDefaultStyle);
}
 




/**
 * Adds a <style:style> (OD_Style_Style class).
 * 
 * @param bAutomatic true if the style is an OpenDocument "automatic style".
 * 
 * @return The address of the newly created OD_Style_Style or NULL, if the
 *         specified style is not currently supported (like graphic styles).
 */
OD_Style_Style* OD_Office_Styles::addStyle(const XML_Char** ppAtts,
                                           OD_ElementStack& rElementStack) {

    const XML_Char* pFamily;
    const XML_Char* pName;
    OD_Style_Style* pStyle;
    bool ok = true;
    bool bOnContentStream;
    
    if (rElementStack.hasElement("office:document-content")) {
        bOnContentStream = true;
    } else {
        bOnContentStream = false;
    }
    
    
    pFamily = UT_getAttribute("style:family", ppAtts);
    pName = UT_getAttribute("style:name", ppAtts);
    UT_ASSERT(pFamily);
    UT_ASSERT(pName);
    
    if (bOnContentStream) {
    
        if(!UT_strcmp(pFamily, "text")) {
            
            pStyle = m_textStyleStyles_contentStream.pick(pName);
            if (pStyle == NULL) {
                pStyle = new OD_Style_Style(rElementStack);
                ok = m_textStyleStyles_contentStream.insert(pName, pStyle);                
            }
            
        } else if(!UT_strcmp(pFamily, "paragraph")) {
            
            pStyle = m_paragraphStyleStyles_contentStream.pick(pName);
            if (pStyle == NULL) {
                pStyle = new OD_Style_Style(rElementStack);
                ok = m_paragraphStyleStyles_contentStream.insert(pName, pStyle);
            }
            
        } else if(!UT_strcmp(pFamily, "section")) {
            
            pStyle = m_sectionStyleStyles_contentStream.pick(pName);
            if (pStyle == NULL) {
                pStyle = new OD_Style_Style(rElementStack);
                ok = m_sectionStyleStyles_contentStream.insert(pName, pStyle);
            }
            
        } else if(!UT_strcmp(pFamily, "graphic")) {
            
            pStyle = m_graphicStyleStyles_contentStream.pick(pName);
            if (pStyle == NULL) {
                pStyle = new OD_Style_Style(rElementStack);
                ok = m_graphicStyleStyles_contentStream.insert(pName, pStyle);
            }
            
        } else {
            // We don't recognize it, yet.
            return NULL;
        }
    
    } else {
        
        if(!UT_strcmp(pFamily, "text")) {
            
            pStyle = m_textStyleStyles.pick(pName);
            if (pStyle == NULL) {
                pStyle = new OD_Style_Style(rElementStack);
                ok = m_textStyleStyles.insert(pName, pStyle);
            }
            
        } else if(!UT_strcmp(pFamily, "paragraph")) {
            
            pStyle = m_paragraphStyleStyles.pick(pName);
            if (pStyle == NULL) {
                pStyle = new OD_Style_Style(rElementStack);
                ok = m_paragraphStyleStyles.insert(pName, pStyle);
            }
            
        } else if(!UT_strcmp(pFamily, "section")) {
            
            pStyle = m_sectionStyleStyles.pick(pName);
            if (pStyle == NULL) {
                pStyle = new OD_Style_Style(rElementStack);
                ok = m_sectionStyleStyles.insert(pName, pStyle);
            }
            
        } else if(!UT_strcmp(pFamily, "graphic")) {
            
            pStyle = m_graphicStyleStyles.pick(pName);
            if (pStyle == NULL) {
                pStyle = new OD_Style_Style(rElementStack);
                ok = m_graphicStyleStyles.insert(pName, pStyle);
            }
            
        } else {
            // We don't recognize it, yet.
            return NULL;
        }
        
    }
    
    UT_ASSERT(ok);
    
    return pStyle;   
}





/**
 * Adds a <style:page-layout> (OD_Style_PageLayout class)
 * 
 * @return The address of the newly created OD_Style_PageLayout.
 */
OD_Style_PageLayout* OD_Office_Styles::addPageLayout(const XML_Char** ppAtts,
                                                     OD_ElementStack& rElementStack) {
                               
    const XML_Char* pAttrValue;
    OD_Style_PageLayout* pStyle;
    bool ok;
                             
    pStyle = new OD_Style_PageLayout(rElementStack);
    pAttrValue = UT_getAttribute("style:name", ppAtts);
    ok = m_pageLayoutStyles.insert(pAttrValue, pStyle);
    
    UT_ASSERT(ok);
    
    return pStyle;
}





/**
 * Adds a <style:master-page> (OD_Style_MasterPage class)
 * 
 * @return The address of the newly created OD_Style_MasterPage.
 */
OD_Style_MasterPage* OD_Office_Styles::addMasterPage(const XML_Char** ppAtts,
                                                     PD_Document* pDocument,
                                                     OD_ElementStack& rElementStack) {
                                                        
    const XML_Char* pAttrValue;
    OD_Style_MasterPage* pStyle;
    bool ok;
                             
    pStyle = new OD_Style_MasterPage(pDocument, rElementStack);
    pAttrValue = UT_getAttribute("style:name", ppAtts);
    ok = m_masterPageStyles.insert(pAttrValue, pStyle);
    
    UT_ASSERT(ok);
    
    return pStyle;
}





/**
 * 
 */
OD_Style_Style* OD_Office_Styles::addDefaultStyle(const XML_Char** ppAtts,
                                                  OD_ElementStack& rElementStack) {
    
    const XML_Char* pAttr;
    
    pAttr = UT_getAttribute("style:family", ppAtts);
    UT_ASSERT(pAttr);

    if (!UT_strcmp("paragraph", pAttr)) {
        m_pParagraphDefaultStyle = new OD_Style_Style(rElementStack);
        return m_pParagraphDefaultStyle;
    } else {
        // Not currently supported
        return NULL;
    }
}





/**
 * Links every style with its parent and next ones.
 */
void OD_Office_Styles::_linkStyles() {
    
    _linkStyles(m_textStyleStyles);
    _linkStyles(m_textStyleStyles_contentStream, &m_textStyleStyles);    
    
    _linkParagraphStyles(false);
    _linkParagraphStyles(true);
        
    _linkStyles(m_sectionStyleStyles);
    _linkStyles(m_sectionStyleStyles_contentStream, &m_sectionStyleStyles);
    
    _linkStyles(m_graphicStyleStyles);
    _linkStyles(m_graphicStyleStyles_contentStream, &m_graphicStyleStyles);
    
    _linkMasterStyles();
    _linkListStyles();
}


/**
 * Helper function for linkStyles()
 * 
 * It links, if applicable, each style with its parent and its next style.
 * 
 * By "linking" I mean that a given style will have a pointer to its parent
 * and its next style.
 * 
 * @param rStyles The list of styles.
 * @param pStyles2 A second list with further styles.
 * @param pDefaultStyle A default style. 
 */   
void OD_Office_Styles::_linkStyles(
                        UT_GenericStringMap<OD_Style_Style*>& rStyles,
                        UT_GenericStringMap<OD_Style_Style*>* pStyles2,
                        const OD_Style_Style* pDefaultStyle) {
    
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    UT_uint32 i, count;
    OD_Style_Style* pStyle;
    const OD_Style_Style* pOtherStyle;
    
    pStylesVec = rStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        pStyle = (*pStylesVec)[i];
        
        // Link to its parent style, if there is one.
        if (!pStyle->getParentStyleName().empty()) {
            
            pOtherStyle = rStyles.pick(
                pStyle->getParentStyleName().utf8_str());
                
            if (pOtherStyle) {
                pStyle->setParentStylePointer(pOtherStyle);
                
            } else if (pStyles2) {
                // It's not on the first list. Let's try to find it on the
                // second one.
                 pOtherStyle = pStyles2->pick(
                    pStyle->getParentStyleName().utf8_str());
                    
                if (pOtherStyle) {
                    pStyle->setParentStylePointer(pOtherStyle);
                }
            }
            
            if (!pOtherStyle) {
                // Neither style lists have it.
                
                if (pDefaultStyle) {
                    pStyle->setParentStylePointer(pDefaultStyle);
                } else {
                    // We don't have this style!
                    UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
                }
            }
                
        }
        
        // Link to its next style, if there is one.
        if (!pStyle->getNextStyleName().empty()) {
            
            pOtherStyle = rStyles.pick(
                pStyle->getNextStyleName().utf8_str());
                
            if (pOtherStyle) {
                pStyle->setNextStylePointer(pOtherStyle);
            } else if (pStyles2) {
                // It's not on the first list. Let's try to find it on the
                // second one.
                
                pOtherStyle = pStyles2->
                                pick(pStyle->getNextStyleName().utf8_str());
                
                if (pOtherStyle) {
                    pStyle->setNextStylePointer(pOtherStyle);
                }
            }
            
            if (!pOtherStyle) {
                // We don't have this style!
                UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
            }
        }
    }
}


/**
 * Link the master styles to their page layouts
 */
void OD_Office_Styles::_linkMasterStyles() {
    
    UT_GenericVector<OD_Style_MasterPage*>* pMasterStylesVec;
    UT_uint32 i, count;
    OD_Style_MasterPage* pMasterStyle;
    OD_Style_PageLayout* pLayout;
    
    pMasterStylesVec = m_masterPageStyles.enumerate();
    UT_ASSERT(pMasterStylesVec);
    
    count = pMasterStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        pMasterStyle = (*pMasterStylesVec)[i];
        
        pLayout = m_pageLayoutStyles.pick(pMasterStyle->getLayoutName().utf8_str());
        UT_ASSERT(pLayout);
        
        pMasterStyle->setLayoutStylePointer(pLayout);
    }
}


/**
 * Link list level styles to the text styles that they refer to.
 */
void OD_Office_Styles::_linkListStyles() {
    
    UT_uint32 i, j, count, count2;
    UT_GenericVector<OD_Style_List*>* pStylesVec;
    OD_ListLevelStyle* pLevelStyle;
    OD_Style_List* pListStyle;
    const OD_Style_Style* pStyle;
    
    pStylesVec = m_listStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {

        pListStyle = (*pStylesVec)[i];
        UT_ASSERT(pListStyle);
        
        count2 = pListStyle->getLevelCount();

        // List levels start from 1.        
        for (j=1; j<=count2; j++) {
            pLevelStyle = pListStyle->getLevelStyle(j);
            
            pStyle = this->getTextStyle(
                pLevelStyle->getTextStyleName()->utf8_str(), false);
            pLevelStyle->setTextStyle(pStyle);
        }
    }
}


/**
 * 
 */
void OD_Office_Styles::_defineAbiStyles(PD_Document* pDocument) const {
    
    UT_uint32 i, count;
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    UT_GenericVector<OD_Style_List*>* pListVec;
    bool ok;
    
    m_pParagraphDefaultStyle->defineAbiStyle(pDocument);
    
    pStylesVec = m_textStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->defineAbiStyle(pDocument);
    }
    
    
    pStylesVec = m_paragraphStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->defineAbiStyle(pDocument);
    }
    
    // AbiWord doesn't have section styles.

    // All styles defined on the content stream are automatic, so, I'm not
    // defining them.


    pListVec = m_listStyles.enumerate();
    UT_ASSERT(pListVec);
    
    count = pListVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pListVec)[i]->defineAbiList(pDocument);
    }
    
    if (count > 0) {
        ok = pDocument->fixListHierarchy();
        UT_ASSERT(ok);
    }
    
    




    // I will just use the first master page style.
    UT_GenericVector<OD_Style_MasterPage*>* pMasterStylesVec;
    
    pMasterStylesVec = m_masterPageStyles.enumerate();
    
    (*pMasterStylesVec)[0]->definePageSizeTag(pDocument);
}





/**
 * 
 */
void OD_Office_Styles::_buildAbiPropsAttrString() {
    
    UT_uint32 i, count;
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    UT_GenericVector<OD_Style_List*>* pListVec;
    
    UT_ASSERT(m_pParagraphDefaultStyle);
    m_pParagraphDefaultStyle->buildAbiPropsAttrString();
    
    
    pStylesVec = m_textStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
    
    
    pStylesVec = m_paragraphStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
    
    
    pStylesVec = m_sectionStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
    
    
    
    
    
    pStylesVec = m_textStyleStyles_contentStream.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
    
    
    pStylesVec = m_paragraphStyleStyles_contentStream.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
    
    
    pStylesVec = m_sectionStyleStyles_contentStream.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
    
    
    
    
    
    
    
    pListVec = m_listStyles.enumerate();
    UT_ASSERT(pListVec);
    
    count = pListVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pListVec)[i]->buildAbiPropertiesString();
    }
}





/**
 * Fix any problems encountered on the added styles
 */
void OD_Office_Styles::_fixStyles() {
    // Problem 1: We can't have styles without properties
    //
    // The "Standard" paragraph style usually comes empty
    // (I have never seen otherwise)
    
    UT_uint32 i, count;
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    OD_Style_Style* pStyle;
    bool noneFound;
    
    do {
        pStylesVec = m_paragraphStyleStyles.enumerate();
        UT_ASSERT(pStylesVec);
        
        noneFound = true;
        count = pStylesVec->getItemCount();
        for (i=0; i<count; i++) {
            if ( !((*pStylesVec)[i]->hasProperties()) ) {
                pStyle = (*pStylesVec)[i];
                i=count;
                noneFound = false;
            }
        }
        
        if (!noneFound) {
            _removeParagraphStyleStyle(pStyle, false);
        }
    } while (!noneFound);
    
    
    
    do {
        pStylesVec = m_paragraphStyleStyles_contentStream.enumerate();
        UT_ASSERT(pStylesVec);
        
        noneFound = true;
        count = pStylesVec->getItemCount();
        for (i=0; i<count; i++) {
            if ( !((*pStylesVec)[i]->hasProperties()) ) {
                pStyle = (*pStylesVec)[i];
                i=count;
                noneFound = false;
            }
        }
        
        if (!noneFound) {
            _removeParagraphStyleStyle(pStyle, true);
        }
    } while (!noneFound);
    
    
    //////
    // Do the same for text styles
    
    do {
        pStylesVec = m_textStyleStyles.enumerate();
        UT_ASSERT(pStylesVec);
        
        noneFound = true;
        count = pStylesVec->getItemCount();
        for (i=0; i<count; i++) {
            if ( !((*pStylesVec)[i]->hasProperties()) ) {
                pStyle = (*pStylesVec)[i];
                i=count;
                noneFound = false;
            }
        }
        
        if (!noneFound) {
            _removeTextStyleStyle(pStyle, false);
        }
    } while (!noneFound);
    
    
    do {
        pStylesVec = m_textStyleStyles_contentStream.enumerate();
        UT_ASSERT(pStylesVec);
        
        noneFound = true;
        count = pStylesVec->getItemCount();
        for (i=0; i<count; i++) {
            if ( !((*pStylesVec)[i]->hasProperties()) ) {
                pStyle = (*pStylesVec)[i];
                i=count;
                noneFound = false;
            }
        }
        
        if (!noneFound) {
            _removeTextStyleStyle(pStyle, true);
        }
    } while (!noneFound);
}





/**
 * 
 */
void OD_Office_Styles::_removeStyleStyle(OD_Style_Style* pRemovedStyle,
                    bool bOnContentStream,
                    UT_GenericStringMap<OD_Style_Style*>& rStyles,
                    UT_GenericStringMap<OD_Style_Style*>& rStyles_contentStream,
                    OD_Style_Style* pDefaultStyle) {
    
    UT_uint32 i, count;
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    OD_Style_Style* pStyle;
    UT_UTF8String styleName;
    UT_UTF8String replacementName;
    bool ok;


    _findSuitableReplacement(replacementName, pRemovedStyle, bOnContentStream,
        rStyles, rStyles_contentStream, pDefaultStyle);
    
    // Remove the style itself
    if (bOnContentStream) {
        rStyles_contentStream.remove(
            pRemovedStyle->getName().utf8_str(), NULL);
            
        ok = m_removedStyleStyles_contentStream.ins(pRemovedStyle->getName(),
                                                   replacementName);
    } else {
        rStyles.remove(pRemovedStyle->getName().utf8_str(), NULL);
        ok = m_removedStyleStyles.ins(pRemovedStyle->getName(), replacementName);
    }
    
    UT_ASSERT(ok);



    // Fix all references to it.
    // Note that automatic styles can't refer to each other.
    
    if (pRemovedStyle->isAutomatic()) {
        // It's an automatic style, nobody have references him.
        return;
    }
    
    if (!UT_strcmp(replacementName.utf8_str(), "<NULL>")) {
        replacementName.clear();
    }
    
    // Some automatic styles defined on the content stream may have
    // references to this common style.
    pStylesVec = rStyles_contentStream.enumerate();

    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        if ((*pStylesVec)[i]->getParentName() == pRemovedStyle->getName()) {
            (*pStylesVec)[i]->setParentName(replacementName);
        }
        
        if ((*pStylesVec)[i]->getNextStyleName() == pRemovedStyle->getName()) {
            (*pStylesVec)[i]->setNextStyleName(replacementName);
        }
    }
    
    // Now fix references from the styles defined on the styles stream.
    pStylesVec = rStyles.enumerate();

    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        if ((*pStylesVec)[i]->getParentName() == pRemovedStyle->getName()) {
            (*pStylesVec)[i]->setParentName(replacementName);
        }
        
        if ((*pStylesVec)[i]->getNextStyleName() == pRemovedStyle->getName()) {
            (*pStylesVec)[i]->setNextStyleName(replacementName);
        }
    }

}


/**
 * Returns the specified paragraph style
 * 
 * @param pStyleName The name of the style wanted.
 */
const OD_Style_Style* OD_Office_Styles::getParagraphStyle(
                                             const XML_Char* pStyleName,
                                             bool bOnContentStream) {
                                                
    OD_Style_Style* pStyle = NULL;
    
    // Is it the default style?
    if (!UT_strcmp(m_pParagraphDefaultStyle->getName().utf8_str(), pStyleName)) {
        pStyle = m_pParagraphDefaultStyle;
    }
    
    if (!pStyle) {
        // It's not the default style. Let's search our style lists.
        
        if (bOnContentStream) {
            pStyle = m_paragraphStyleStyles_contentStream.pick(pStyleName);
            if (!pStyle) {
                // Should be a regular style (not automatic).
                pStyle = m_paragraphStyleStyles.pick(pStyleName);
            }
        } else {
            pStyle = m_paragraphStyleStyles.pick(pStyleName);
        }
    }
    
    if (!pStyle) {
        // We haven't found it. Have we removed it (done on _fixStyles())?
        
        const UT_UTF8String* pReplacementName;
        
        if (bOnContentStream) {
            pReplacementName = m_removedStyleStyles_contentStream[pStyleName];
            
            if (pReplacementName == NULL) {
                pReplacementName = m_removedStyleStyles[pStyleName];
            }
        } else {
            pReplacementName = m_removedStyleStyles[pStyleName];
        }
        
        if (pReplacementName) {
            // We will send back its replacement.
            return this->getParagraphStyle(pReplacementName->utf8_str(),
                                           bOnContentStream);
        } else {
            // This style never existed.
            pStyle = NULL;
        }
        
    }
    
    return pStyle;
}


/**
 * 
 */
const OD_Style_Style* OD_Office_Styles::getTextStyle(const XML_Char* pStyleName,
                                              bool bOnContentStream) const {
    OD_Style_Style* pStyle;
    
    if (bOnContentStream) {
        pStyle = m_textStyleStyles_contentStream.pick(pStyleName);
        
        if (!pStyle) {
            // Should be a regular style (not automatic).
            pStyle = m_textStyleStyles.pick(pStyleName);
        }
        
        return pStyle;
    } else {
        return m_textStyleStyles.pick(pStyleName);
    }
}
    

/**
 * 
 */ 
const OD_Style_Style* OD_Office_Styles::getSectionStyle(const XML_Char* pStyleName,
                                                 bool bOnContentStream) const {
    OD_Style_Style* pStyle;
    
    if (bOnContentStream) {
        pStyle = m_sectionStyleStyles_contentStream.pick(pStyleName);
        
        if (!pStyle) {
            // Should be a regular style (not automatic).
            pStyle = m_sectionStyleStyles.pick(pStyleName);
        }
        
        return pStyle;
    } else {
        return m_sectionStyleStyles.pick(pStyleName);
    }
}


/**
 * 
 */
const OD_Style_Style* OD_Office_Styles::getGraphicStyle(const XML_Char* pStyleName,
                                          bool bOnContentStream) const {
    OD_Style_Style* pStyle;
    
    if (bOnContentStream) {
        pStyle = m_graphicStyleStyles_contentStream.pick(pStyleName);
        
        if (!pStyle) {
            // Should be a regular style (not automatic).
            pStyle = m_graphicStyleStyles.pick(pStyleName);
        }
        
        return pStyle;
    } else {
        return m_graphicStyleStyles.pick(pStyleName);
    }
}


/**
 * Adds a list style (<text:list-style>)
 */
OD_Style_List* OD_Office_Styles::addList(const XML_Char** ppAtts,
                                         OD_ElementStack& rElementStack) {
    bool ok;
    const XML_Char* pAttrValue = NULL;
    OD_Style_List* pStyle = NULL;

    pStyle = new OD_Style_List(rElementStack);
    pAttrValue = UT_getAttribute("style:name", ppAtts);
                                            
    ok = m_listStyles.insert(pAttrValue, pStyle);
            
    UT_ASSERT(ok);
    
    return pStyle;
}


/**
 * Finds a suitable replacement for the style that will be removed.
 * 
 * A suitable replacement is a parent style that has properties or, if none is
 * found, the default paragraph style.
 * 
 * @param rReplacementName Will receive the name of the replacement style.
 * @param pRemovedStyle The style that will be removed.
 */
void OD_Office_Styles::_findSuitableReplacement(UT_UTF8String& rReplacementName,
                    const OD_Style_Style* pRemovedStyle,
                    bool bOnContentStream,
                    UT_GenericStringMap<OD_Style_Style*>& rStyles,
                    UT_GenericStringMap<OD_Style_Style*>& rStyles_contentStream,
                    OD_Style_Style* pDefaultStyle) {

    // Test for a "dead-end"
    if (pRemovedStyle->getParentStyleName().empty()) {
        
        if (pDefaultStyle) {
            // Pretty simple. We use the default style.
            if (*(pRemovedStyle->getFamily()) == "paragraph") {
                // AbiWord uses "Normal" as the name of its default style.
                rReplacementName = "Normal";
            } else {
                rReplacementName = pDefaultStyle->getName();
            }
        } else {
            // We have no choice. There will be no substitute for this style.
            rReplacementName = "<NULL>";
        }
        
        return;
    }

    OD_Style_Style* pStyle;
    
    if (bOnContentStream) {
        pStyle = rStyles_contentStream.pick(
                    pRemovedStyle->getParentStyleName().utf8_str());
        
        if (!pStyle) {
            // Must be a regular style, defined on the Styles stream.
            pStyle = rStyles.pick(
                        pRemovedStyle->getParentStyleName().utf8_str());
        }
        
    } else {
        pStyle = rStyles.pick(
                    pRemovedStyle->getParentStyleName().utf8_str());
    }
    
    
    if (pStyle) {
        if (pStyle->hasProperties()) {
            // Alright, we've found it.
            rReplacementName = pStyle->getName();
        } else {
            // Let's look deeper
            _findSuitableReplacement(rReplacementName, pStyle, bOnContentStream,
                rStyles, rStyles_contentStream, pDefaultStyle);
        }
    } else {
        const UT_UTF8String* pString;
        // Was this parent already removed?
        if (bOnContentStream) {
            pString = m_removedStyleStyles_contentStream[
                                    pRemovedStyle->getParentStyleName()];
        }
        
        if (!pStyle) {
            pString = m_removedStyleStyles[pRemovedStyle->getParentStyleName()];
        }
        
        if(pString) {
            rReplacementName = *pString;
        } else {
            // I give up...
            if (pDefaultStyle) {
                // Pretty simple. We use the default style.
                if (*(pRemovedStyle->getFamily()) == "paragraph") {
                    // AbiWord uses "Normal" as the name of its default style.
                    rReplacementName = "Normal";
                } else {
                    rReplacementName = pDefaultStyle->getName();
                }
            } else {
                // We have no choice. There will be no substitute for this style.
                rReplacementName = "<NULL>";
            }
        }
    }
}


/**
 * 
 */
void OD_Office_Styles::_linkParagraphStyles(bool bOnContentStream) {
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    UT_uint32 i, count;
    OD_Style_Style* pStyle;
    const OD_Style_Style* pOtherStyle;
    
    if (bOnContentStream) {
        pStylesVec = m_paragraphStyleStyles_contentStream.enumerate();
    } else {
        pStylesVec = m_paragraphStyleStyles.enumerate();
    }
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        pStyle = (*pStylesVec)[i];
        
        // Link to its parent style, if there is one.
        if (!pStyle->getParentStyleName().empty()) {
            pOtherStyle = this->getParagraphStyle(
                                pStyle->getParentStyleName().utf8_str(),
                                bOnContentStream);
                                
            if (pOtherStyle) {
                pStyle->setParentStylePointer(pOtherStyle);
            } else {
                // That's a problem. To avoid an abort, let's replace it with
                // the default style.
                pStyle->setParentStylePointer(m_pParagraphDefaultStyle);
            }
        }
        
        // Link to its next style, if there is one.
        if (!pStyle->getNextStyleName().empty()) {
            pOtherStyle = this->getParagraphStyle(
                                pStyle->getNextStyleName().utf8_str(),
                                bOnContentStream);
                                
            if (pOtherStyle) {
                pStyle->setNextStylePointer(pOtherStyle);
            } else {
                // That's a problem. To avoid an abort, let's replace it with
                // the default style.
                pStyle->setNextStylePointer(m_pParagraphDefaultStyle);
            }
        }
    }
}
