/* AbiSource
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */

// Class definition include
#include "OD_Frame_ListenerState.h"

// Internal includes
#include "OD_ListenerStateAction.h"
#include "OD_ElementStack.h"
#include "OD_Style_Style.h"
#include "OD_StartTag.h"
#include "OD_Office_Styles.h"
#include "OD_Abi_Data.h"

// AbiWord includes
#include <pt_Types.h>
#include <pd_Document.h>


/**
 * Constructor
 */
OD_Frame_ListenerState::OD_Frame_ListenerState(PD_Document* pDocument,
    OD_Office_Styles* pStyles,
    OD_Abi_Data& rAbiData,
    OD_ElementStack& rElementStack) :
        OD_ListenerState("Frame", rElementStack),
        m_pAbiDocument(pDocument),
        m_rAbiData(rAbiData),
	    m_pStyles(pStyles),
        m_parsedFrameStartTag(false)
{
    if (m_rElementStack.hasElement("office:document-content")) {
        m_bOnContentStream = true;
    } else {
        m_bOnContentStream = false;
    }
}


/**
 * 
 */
void OD_Frame_ListenerState::startElement (const XML_Char* pName,
                                           const XML_Char** ppAtts,
                                           OD_ListenerStateAction& rAction) {

    if (!m_waitingEndElement.empty()) {
        // Do nothing.
        
    } else if (!UT_strcmp(pName, "draw:frame")) {
        if (m_parsedFrameStartTag) {
            // It's a nested frame.
            rAction.pushState("Frame");
        } else {
            m_parsedFrameStartTag = true;
        }
        
    } else if (!UT_strcmp(pName, "draw:image")) {
        _drawImage(ppAtts);
        
    } else if (!UT_strcmp(pName, "draw:text-box")) {
        if (m_rElementStack.hasElement("draw:text-box")) {
            // AbiWord doesn't support nested text boxes.
            // Let's ignore that one
            m_waitingEndElement = "draw:text-box";
        } else {
            // We are going to receive text content.
            rAction.pushState("TextContent");
        }
    }
}


/**
 * 
 */                
void OD_Frame_ListenerState::endElement (const XML_Char* pName,
                                         OD_ListenerStateAction& rAction) {

    if (!m_waitingEndElement.empty()) {
        
        if ( !UT_strcmp(m_waitingEndElement.utf8_str(), pName) ) {
            // The wait has ended.
            m_waitingEndElement.clear();
        }
        
    } else if (!UT_strcmp(pName, "draw:frame")) {
        // We're done.
        rAction.popState();
    }
}


/**
 * 
 */
void OD_Frame_ListenerState::charData (const XML_Char* pBuffer, int length) {
    // Nothing yet...
}


/**
 * 
 * @param The attributes of a <draw:image> element.
 */
void OD_Frame_ListenerState::_drawImage (const XML_Char** ppAtts)
{
    const XML_Char* pStyleName;
    const OD_Style_Style* pGraphicStyle;
    const UT_UTF8String* pWrap;
    UT_String dataId; // id of the data item that contains the image.
    
    
    //
    // Adds a reference to the added data item according to wrap mode, etc.
    //
   
    pStyleName = m_rElementStack.getStartTag(0)->getAttributeValue("draw:style-name");
    UT_ASSERT(pStyleName);
    
    pGraphicStyle = m_pStyles->getGraphicStyle(pStyleName, m_bOnContentStream);
    UT_ASSERT(pGraphicStyle);
    
    pWrap = pGraphicStyle->getWrap(false);
    
    if ( !UT_strcmp(pWrap->utf8_str(), "none") ||
         pWrap->empty()) {
        // In-line wrapping.
        // No frames are used on AbiWord for in-line wrapping.
        // It uses a <image> tag right in the paragraph text.
        
        const XML_Char* pWidth;
        const XML_Char* pHeight;
        
        m_rAbiData.addImageDataItem(dataId, ppAtts);
       
        const XML_Char* attribs[5];
        UT_String propsBuffer;
        
        pWidth = m_rElementStack.getStartTag(0)->getAttributeValue("svg:width");
        UT_ASSERT(pWidth);
        
        pHeight = m_rElementStack.getStartTag(0)->getAttributeValue("svg:height");
        UT_ASSERT(pHeight);  
        
        UT_String_sprintf(propsBuffer, "width:%s; height:%s", pWidth, pHeight);
        
        attribs[0] = "props";
        attribs[1] = propsBuffer.c_str();
        attribs[2] = "dataid";
        attribs[3] = dataId.c_str();
        attribs[4] = 0;
    
        if (!m_pAbiDocument->appendObject (PTO_Image, attribs)) {
            UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
        }
        
    } else {
        // We define a frame with the image in it.
        
        if (m_rElementStack.hasElement("draw:text-box")) {
            // AbiWord can't have nested frames.
            return;
        }
        
        const XML_Char* attribs[5];
        const XML_Char* pVal;
        UT_UTF8String props;
        
        props = "frame-type:image";
            
        if ( !UT_strcmp(pWrap->utf8_str(), "run-through")) {
            // Floating wrapping.
            props += "; wrap-mode:above-text";
            
        } else if ( !UT_strcmp(pWrap->utf8_str(), "left")) {
            props += "; wrap-mode:wrapped-to-left";
            
        } else if ( !UT_strcmp(pWrap->utf8_str(), "right")) {
            props += "; wrap-mode:wrapped-to-right";
            
        } else if ( !UT_strcmp(pWrap->utf8_str(), "parallel")) {
            props += "; wrap-mode:wrapped-both";
            
        } else {
            // Unsupported. We're doomed.
            UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
            
            // Let's put an arbitrary wrap mode to avoid an error.
            props += "; wrap-mode:wrapped-both";
        }
        
        pVal = m_rElementStack.getStartTag(0)->getAttributeValue("text:anchor-type");
        UT_ASSERT(pVal);
        props += "; position-to:";
        if (!UT_strcmp(pVal, "paragraph")) {
            props += "block-above-text";
            
            pVal = m_rElementStack.getStartTag(0)->getAttributeValue("svg:x");
            UT_ASSERT(pVal);
            props += "; xpos:";
            props += pVal;
            
            pVal = m_rElementStack.getStartTag(0)->getAttributeValue("svg:y");
            UT_ASSERT(pVal);
            props += "; ypos:";
            props += pVal;
            
        } else if (!UT_strcmp(pVal, "page")) {
            props += "page-above-text";
            
            pVal = m_rElementStack.getStartTag(0)->getAttributeValue("svg:x");
            UT_ASSERT(pVal);
            props += "; frame-page-xpos:";
            props += pVal;
            
            pVal = m_rElementStack.getStartTag(0)->getAttributeValue("svg:y");
            UT_ASSERT(pVal);
            props += "; frame-page-ypos:";
            props += pVal;
            
        } else {
            // Unsupported. We're doomed.
            return;
        }
        
        
        pVal = m_rElementStack.getStartTag(0)->getAttributeValue("svg:width");
        UT_ASSERT(pVal);
        props += "; frame-width:";
        props += pVal;
        
        pVal = m_rElementStack.getStartTag(0)->getAttributeValue("svg:height");
        UT_ASSERT(pVal);
        props += "; frame-height:";
        props += pVal;
        
        
        // Avoid having frame border lines.
        props += "; bot-style:none; left-style:none; right-style:none; top-style:none";
        
        
        m_rAbiData.addImageDataItem(dataId, ppAtts);
        
        attribs[0] = "strux-image-dataid";
        attribs[1] = dataId.c_str();
        attribs[2] = "props";
        attribs[3] = props.utf8_str();
        attribs[4] = 0;
        
        if(!m_pAbiDocument->appendStrux(PTX_SectionFrame, attribs)) {
            UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
        }
        
        if(!m_pAbiDocument->appendStrux(PTX_EndFrame, NULL)) {
            UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
        }
    }

}
