/* AbiSource Program Utilities
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */
 
// Class definition include
#include "OD_ElementStack.h"
 
// Internal includes
#include "OD_StartTag.h"

// AbiWord includes
#include <ut_string.h>


/**
 * Constructor
 */
OD_ElementStack::OD_ElementStack() :
                   m_pStartTags(NULL),
                   m_stackSize(0) {
            
}


/**
 * Destructor
 */
OD_ElementStack::~OD_ElementStack() {

    UT_VECTOR_PURGEALL(OD_StartTag*, (*m_pStartTags));
    DELETEP(m_pStartTags);
}


/**
 * Must be the last command called by the starElement method of the listener
 * class.
 */
void OD_ElementStack::startElement (const XML_Char* pName,
                                                 const XML_Char** ppAtts) {

    OD_StartTag* pStartTag;

    if (!m_pStartTags) {
        m_pStartTags = new UT_GenericVector <OD_StartTag*> (10, 10);
    }

    if (m_stackSize == m_pStartTags->getItemCount()) { 
        
        pStartTag = new OD_StartTag();
        m_pStartTags->push_back(pStartTag);
        
    } else if (m_stackSize < m_pStartTags->getItemCount()) {
        
        pStartTag = (*m_pStartTags)[m_stackSize];
        
    } else {
        UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
    }
    
    pStartTag->set(pName, ppAtts);
    
    m_stackSize++;
}


/**
 * Must be the last command called by the endElement method of the listener
 * class.
 */
void OD_ElementStack::endElement (const XML_Char* pName) {
    UT_ASSERT(m_pStartTags != NULL);
    m_stackSize--;
}


/**
 * @param level 0 is the immediate parent, 1 is the parent of the parent
 *              and so on.
 * 
 * On the startElement method, level 0 is the parent start tag.
 * On the endElement method, level 0 is the corresponding start tag.
 */
const OD_StartTag* OD_ElementStack::getStartTag(UT_uint32 level) {
    
    if (m_pStartTags) {
        if (m_stackSize > level) {
            // The level is counted from the top of the vector down to the bottom
            // so, level 0 is m_pStartTags[lastIndex] and
            // level max is m_pStartTags[0]
            return (*m_pStartTags)[m_stackSize - (level+1)];
        } else {
            return NULL;
        }
    } else {
        return NULL;
    }
}



/**
 * @return True if at least one of the stack elements has the specified name.
 */
bool OD_ElementStack::hasElement(const XML_Char* pName) const {
    UT_uint32 i, count;
    OD_StartTag* pStartTag;
    
    count = m_pStartTags->getItemCount();
    
    for (i=0; i<count; i++) {
        pStartTag = (*m_pStartTags)[i];
        if (!UT_strcmp(pStartTag->getName(), pName)) {
            return true;
        }
    }
    
    // If the execution reached this line it's because no match was found.
    return false;
}
