/* AbiSource
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */


#ifndef _OD_STYLE_STYLE_H_
#define _OD_STYLE_STYLE_H_

// Internal includes
#include "OD_ListenerState.h"

// AbiWord includes
#include <ut_string_class.h>

// AbiWord classes
class PD_Document;


/**
 * An OpenDocument regular style (<style:style>).
 */
class OD_Style_Style : public OD_ListenerState {
    
public:

    OD_Style_Style(bool bAutomatic);

    void startElement(const XML_Char* pName, const XML_Char** ppAtts,
                      OD_ListenerStateAction& rAction);
                      
    void endElement(const XML_Char* pName, OD_ListenerStateAction& rAction);
    
    void charData (const XML_Char* pBuffer, int length) {}
    
    
    inline const UT_UTF8String& getDisplayName() const {return m_displayName;}
    
    /**
     * Defines an AbiWord style that is equivalent to this
     * OpenDocument style.
     * 
     * Called by text and paragraph styles.
     * 
     * @param pDocument The AbiWord document on which the style will be defined.
     */
    void defineAbiStyle(PD_Document* pDocument);
    
    
    /**
     * Builds the AbiWord "props" attribute value that describes this
     * Style.
     */
    void buildAbiPropsAttrString();
    
    inline const UT_UTF8String& getAbiPropsAttrString() const {
        return m_abiPropsAttr;
    }
    
    
    inline const UT_UTF8String& getParentStyleName() const {
        return m_parentStyleName;
    }
    
    inline void setParentStylePointer(const OD_Style_Style* pParentStyle) {
        m_pParentStyle = pParentStyle;        
    }
    
    inline void setNextStylePointer(const OD_Style_Style* pNextStyle) {
        m_pNextStyle = pNextStyle;
    }
    
    inline bool getPageBreakBefore() const {return m_bPageBreakBefore;}
    inline bool getColBreakBefore() const {return m_bColBreakBefore;}
    
    inline const UT_UTF8String& getName() const {return m_name;}
    
    inline const UT_UTF8String& getParentName() const {return m_parentStyleName;}
    inline void setParentName(const char* pName) {m_parentStyleName.assign(pName);}

    inline const UT_UTF8String& getNextStyleName() const {return m_nextStyleName;}
    inline void setNextStyleName(const char* pName) {m_nextStyleName.assign(pName);}
 
    inline bool hasProperties() const {
        return m_bColBreakBefore ||
               m_bPageBreakBefore ||
               !m_lineHeight.empty() ||
               !m_align.empty() ||
               !m_widows.empty() ||    
               !m_orphans.empty() || 
               !m_marginLeft.empty() ||
               !m_marginRight.empty() ||
               !m_marginTop.empty() ||
               !m_marginBottom.empty() ||       
               !m_bgcolor.empty() ||
               !m_keepWithNext.empty() ||
            
               !m_color.empty() ||
               !m_textDecoration.empty() ||
               !m_textPos.empty() ||
               !m_fontName.empty() ||
               !m_fontSize.empty() ||
               !m_lang.empty() ||
               !m_fontStyle.empty() ||
               !m_fontWeight.empty() ||
            
               !m_columns.empty();
    }
    
    inline bool isAutomatic() const {return m_bAutomatic;}
 
private:

    // <style:style />
    void _parse_style_style(const XML_Char** ppAtts);
    
    // <style:paragraph-properties />
    void _parse_style_paragraphProperties(const XML_Char** ppProps);
    
    // <style:text-properties />
    void _parse_style_textProperties(const XML_Char** ppProps);
    
    // <style:section-properties />
    void _parse_style_sectionProperties(const XML_Char** ppProps);
    
    
    // true if it is an OpenDocument automatic style.
    // ie., it's defined inside a <office:automatic-styles> element.
    bool m_bAutomatic;
    
    const OD_Style_Style* m_pParentStyle;
    const OD_Style_Style* m_pNextStyle;
    
    UT_UTF8String m_abiPropsAttr;
    

    // <attribute name="style:name">
    UT_UTF8String m_name;

    // <attribute name="style:display-name"> (optional)
    // If this attribute is not present, the display name equals the style name.
    // In AbiWord, maps to the "name" attribute.
    UT_UTF8String m_displayName;
    
    // Maps to the "type" attribute.
    // OpenDocument | AbiWord
    // "character"  - "C"
    // "paragraph"  - "P"
    // "section"    - none (AbiWord don't have section styles)
    UT_UTF8String m_family;
    
    // <attribute name="style:parent-style-name"> (optional)
    // If a parent style is not specified, a default parent style defined by
    // the application is used.
    //
    // In AbiWord, maps to the "basedon" attribute.
    UT_UTF8String m_parentStyleName;
    
    // <attribute name="style:next-style-name">
    // By default, the current style is used as the next style.
    // In AbiWord, maps to the "followedby" attribute.
    UT_UTF8String m_nextStyleName;
    
    // <attribute name="style:list-style-name"> (optional)
    //
    // Is only applied to headings and to paragraphs that are contained in a
    // list, where the list does not specify a list style itself, and the list
    // has no list style specification for any of its parents.
    //
    // Maps to AbiWord, but not directly.
    UT_UTF8String m_listStyleName;
    
    // <attribute name="style:master-page-name"> (optional)
    //
    // If this attribute is associated with a style, a page break is inserted
    // when the style is applied and the specified master page is applied to the
    // preceding page.
    // This attribute is ignored if it is associated with a paragraph style that
    // is applied to a paragraph within a table.
    //
    // Maps to AbiWord, but not directly.
    UT_UTF8String m_MasterPageName;
    
    

    // <style:paragraph-properties> attributes
    // These goes inside the Abi "props" attribute
    bool m_bColBreakBefore;
    bool m_bPageBreakBefore;    
    UT_UTF8String m_lineHeight;
    UT_UTF8String m_align;
    UT_UTF8String m_widows;    
    UT_UTF8String m_orphans; 
    UT_UTF8String m_marginLeft;
    UT_UTF8String m_marginRight;
    UT_UTF8String m_marginTop;
    UT_UTF8String m_marginBottom;       
    UT_UTF8String m_bgcolor;
    UT_UTF8String m_keepWithNext;
    
    // <style:text-properties />
    // These goes inside the Abi "props" attribute
    UT_UTF8String m_color;
    UT_UTF8String m_textDecoration;
    UT_UTF8String m_textPos;
    UT_UTF8String m_fontName;
    UT_UTF8String m_fontSize;
    UT_UTF8String m_lang;
    UT_UTF8String m_fontStyle;
    UT_UTF8String m_fontWeight;
    
    // <style:section-properties>
    // These goes inside the Abi "props" attribute
    UT_UTF8String m_columns;
};

#endif //_OD_STYLE_STYLE_H_
