/* AbiSource
 * 
 * Copyright (C) 2002 Dom Lachowicz <cinamod@hotmail.com>
 * Copyright (C) 2004 Robert Staudinger <robsta@stereolyzer.net>
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */

// Class definition include
#include "OD_TextContent_ListenerState.h"

// Internal includes
#include "OD_Office_Styles.h"
#include "OD_Style_Style.h"
#include "OD_Style_MasterPage.h"
#include "OD_ListenerStateAction.h"

// AbiWord includes
#include <ut_misc.h>
#include <pd_Document.h>
#include <ie_impGraphic.h>
#include <fg_GraphicRaster.h>

// External includes
#include <glib-object.h>
#include <gsf/gsf-input-stdio.h>
#include <gsf/gsf-infile.h>
#include <gsf/gsf-infile-zip.h>




/**
 * Constructor
 */
OD_TextContent_ListenerState::OD_TextContent_ListenerState (
                PD_Document* pDocument,
                GsfInfile* pGsfInfile,
                OD_Office_Styles* pStyles)
                : OD_ListenerState("TextContent"),
                  m_pAbiDocument ( pDocument ),
                  m_pStyles(pStyles),
                  m_pGsfInfile(pGsfInfile),
                  m_bAcceptingText(false),
                  m_bInSection(false),
                  m_imgCnt(0),
                  m_row(0),
                  m_col(0),
                  m_cel(0),
                  m_bInTOC(false),
                  m_bOnContentStream(false)
{
    UT_ASSERT(m_pAbiDocument);
    UT_ASSERT(m_pStyles);
    UT_ASSERT(m_pGsfInfile);
}


/**
 * Called when the XML parser finds a start element tag.
 * 
 * @param pName The name of the element.
 * @param ppAtts The attributes of the parsed start tag.
 */
void OD_TextContent_ListenerState::startElement (const XML_Char* pName,
                                          const XML_Char** ppAtts,
                                          OD_ListenerStateAction& rAction)
{
    
    // if we're inside a TOC, ignore the contents since ODT
    // includes a copy of the TOC
    if (m_bInTOC) {
        return;
    }

    if (!UT_strcmp(pName, "text:section" )) {

        const XML_Char* pStyleName = UT_getAttribute ("text:style-name", ppAtts);
        const OD_Style_Style* pStyle = m_pStyles->getSectionStyle(pStyleName,
                                                        m_bOnContentStream);

        _insureInSection(pStyle->getAbiPropsAttrString().utf8_str());

    } else if (!UT_strcmp(pName, "text:p" ) || !UT_strcmp(pName, "text:h" )) {

        const XML_Char* pStyleName;

        pStyleName = UT_getAttribute ("text:style-name", ppAtts);
        const OD_Style_Style* pStyle;

        if (pStyleName) {
            pStyle = m_pStyles->getParagraphStyle(pStyleName, m_bOnContentStream);
            UT_ASSERT(pStyle);
        } else {
            // Use the default style
            pStyle = m_pStyles->getDefaultParagraphStyle();
        }

        // Do we have an AbiWord section?
        _insureInSection();
        
        // Append an empty paragraph with this one char
        if (pStyle->getPageBreakBefore()) {
            m_pAbiDocument->appendStrux(PTX_Block, NULL);
            UT_UCSChar ucs = (pStyle->getColBreakBefore () ? UCS_VTAB : UCS_FF);
            m_pAbiDocument->appendSpan (&ucs, 1);
        }



        const XML_Char *ppParagraphAtts[3];
        
        if (pStyle->isAutomatic()) {
            // Automatic styles are not defined on the document, so, we
            // just paste its properties.
            ppParagraphAtts[0] = "props";
            ppParagraphAtts[1] = pStyle->getAbiPropsAttrString().utf8_str();
            ppParagraphAtts[2] = 0;
        } else {
            // We refer to the style
            ppParagraphAtts[0] = "style";
            ppParagraphAtts[1] = pStyle->getDisplayName().utf8_str();
            ppParagraphAtts[2] = 0;
        }
        
        m_pAbiDocument->appendStrux(PTX_Block, (const XML_Char**)ppParagraphAtts);
        
        // We now accept text
        m_bAcceptingText = true;

    } else if (!UT_strcmp(pName, "text:s")) {
        // A number of consecutive white-space characters.
        
        const XML_Char* pSpaceCount;
        UT_uint32 spaceCount, i;
        UT_UCS4String string;
        
        pSpaceCount = UT_getAttribute("text:c", ppAtts);
        i = sscanf(pSpaceCount, "%d", &spaceCount);
        UT_ASSERT(i==1);
        
        // TODO: A faster (wiser) implementation can be done, I think. (Daniel d'Andrada)
        string.clear();
        for (i=0; i<spaceCount; i++) {
            string += " ";
        }
        
        // Write the text that has not been written yet.
        // Otherwise the spaces will appear in the wrong position.
        _flush();
        
        m_pAbiDocument->appendSpan(string.ucs4_str(), string.size());
        
    } else if (!UT_strcmp(pName, "text:tab")) {
        // A tab character.

        UT_UCS4String string = "\t";
       
        // Write the text that has not been written yet.
        // Otherwise the spaces will appear in the wrong position.
        _flush();
        
        m_pAbiDocument->appendSpan(string.ucs4_str(), string.size());
        
    } else if (!UT_strcmp(pName, "text:table-of-content")) {
        
        _flush ();
        _insureInBlock(NULL);

        m_pAbiDocument->appendStrux(PTX_SectionTOC, NULL);
        m_pAbiDocument->appendStrux(PTX_EndTOC, NULL);

        m_bInTOC = true;
        
    } else if (!UT_strcmp(pName, "text:span")) {
        // Write all text that is between the last element tag and this
        // <text:span>
        _flush ();

        const XML_Char* pStyleName = UT_getAttribute("text:style-name", ppAtts);
        const OD_Style_Style* pStyle;
        
        if (pStyleName) {
            pStyle = m_pStyles->getTextStyle(pStyleName, m_bOnContentStream);
            UT_ASSERT(pStyle);
        } else {
            // I haven't seen default styles for "text" family.
            UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
        }
        
        if (!pStyle) {
            // If it doens't specify a style it's beacause it does specify a
            // style class an we don't support style classes yet.
            UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
        } else {
        
            const XML_Char* ppStyAttr[3];
            bool ok;
            
            if (pStyle->isAutomatic()) {
                // It goes "hardcoded"
                ppStyAttr[0] = "props";
                ppStyAttr[1] = pStyle->getAbiPropsAttrString().utf8_str();
                ppStyAttr[2] = 0;
            } else {
                ppStyAttr[0] = "style";
                ppStyAttr[1] = pStyle->getDisplayName().utf8_str();
                ppStyAttr[2] = 0;                
            }
            
            _pushInlineFmt(ppStyAttr);
            ok = m_pAbiDocument->appendFmt(&m_vecInlineFmt);
            UT_ASSERT(ok);
        }
        
    } else if (!UT_strcmp(pName, "text:line-break")) {
        
        m_charData += UCS_LF;
        _flush ();
        
    } else if (!UT_strcmp(pName, "text:a")) {
        
        const XML_Char * xlink_atts[3];
        xlink_atts[0] = "xlink:href";
        xlink_atts[1] = UT_getAttribute("xlink:href", ppAtts);
        xlink_atts[2] = 0;
        m_pAbiDocument->appendObject(PTO_Hyperlink, xlink_atts);
        
    } else if (!UT_strcmp(pName, "text:bookmark")) {
        
        _flush ();
        const XML_Char * pName = UT_getAttribute ("text:name", ppAtts);
        _insertBookmark (pName, "start");
        _insertBookmark (pName, "end");
        
    } else if (!UT_strcmp(pName, "draw:image")) {
        
        _flush ();
        _insureInBlock(NULL);
        _insertImage (ppAtts);
        
    } else if (!UT_strcmp(pName, "table:table")) {
        
        _insureInSection(NULL);
        _openTable (ppAtts);
        
    } else if (!UT_strcmp(pName, "table:table-column")) {
        
        _openColumn (ppAtts);
        
    } else if (!UT_strcmp(pName, "table:table-row")) {
        
        _openRow (ppAtts);
        
    } else if (!UT_strcmp(pName, "table:table-cell")) {
        
        _openCell (ppAtts);
        
    } else if (!UT_strcmp(pName, "text:date") ||
            !UT_strcmp(pName, "text:time") ||
            !UT_strcmp(pName, "text:page-number") ||
            !UT_strcmp(pName, "text:page-count") ||
            !UT_strcmp(pName, "text:file-name") ||
            !UT_strcmp(pName, "text:paragraph-count") ||
            !UT_strcmp(pName, "text:word-count") ||
            !UT_strcmp(pName, "text:character-count") ||
            !UT_strcmp(pName, "text:initial-creator") ||
            !UT_strcmp(pName, "text:author-name") ||
            !UT_strcmp(pName, "text:description") ||
            !UT_strcmp(pName, "text:keywords") ||
            !UT_strcmp(pName, "text:subject") ||
            !UT_strcmp(pName, "text:title")) {
                
        _flush ();

        const XML_Char * type = "";
        if(!UT_strcmp(pName, "text:date"))
            type = "date_ddmmyy";
        else if(!UT_strcmp(pName, "text:time"))
            type = "time";
        else if(!UT_strcmp(pName, "text:page-number"))
            type = "page_number";
        else if(!UT_strcmp(pName, "text:page-count"))
            type = "page_count";
        else if(!UT_strcmp(pName, "text:file-name"))
            type = "file_name";
        else if(!UT_strcmp(pName, "text:paragraph-count"))
            type = "para_count";
        else if(!UT_strcmp(pName, "text:word-count"))
            type = "word_count";
        else if(!UT_strcmp(pName, "text:character-count"))
            type = "char_count";
        else if(!UT_strcmp(pName, "text:initial-creator") || !UT_strcmp(pName, "text:author-name"))
            type = "meta_creator";
        else if(!UT_strcmp(pName, "text:description"))
            type = "meta_description";
        else if(!UT_strcmp(pName, "text:keywords"))
            type = "meta_keywords";
        else if(!UT_strcmp(pName, "text:subject"))
            type = "meta_subject";
        else if(!UT_strcmp(pName, "text:title"))
            type = "meta_title";

        const XML_Char *field_fmt[3];
        field_fmt[0] = "type";
        field_fmt[1] = type;
        field_fmt[2] = 0;
        m_pAbiDocument->appendObject(PTO_Field, (const XML_Char**)field_fmt);
        m_bAcceptingText = false;
        
    } else if (!UT_strcmp(pName, "style:header") ||
               !UT_strcmp(pName, "style:footer") ||
               !UT_strcmp(pName, "style:header-left") ||
               !UT_strcmp(pName, "style:footer-left")) {
        // We are inside a header/footer so, there is already a section defined
        // on the AbiWord document.
        m_bInSection = true;
        m_bOnContentStream = false;
        
    } else if (!UT_strcmp(pName, "office:text")) {
        m_bOnContentStream = true;
    }
    
    m_elementStack.startElement(pName, ppAtts);
}


/**
 * Called when an "end of element" tag is parsed (like <myElementName/>)
 * 
 * @param pName The name of the element
 */
void OD_TextContent_ListenerState::endElement (const XML_Char* pName,
                                               OD_ListenerStateAction& rAction)
{
    if (!UT_strcmp(pName, "text:section" )) {

        m_bInSection = false;

    } else if (!UT_strcmp(pName, "text:p" ) || !UT_strcmp(pName, "text:h" )) {
        
        _flush ();
        m_bAcceptingText = false;
        
    } else if (!UT_strcmp(pName, "text:table-of-content")) {
        
        m_bInTOC = false;
        
    } else if (!UT_strcmp(pName, "text:span")) {
        
        _flush ();
        _popInlineFmt();
        m_pAbiDocument->appendFmt(&m_vecInlineFmt);
        
    } else if (!UT_strcmp(pName, "text:a")) {
        
        _flush ();
        m_pAbiDocument->appendObject(PTO_Hyperlink, NULL);
        
    } else if (!UT_strcmp(pName, "table:table")) {
        
        _closeTable ();
        
    } else if (!UT_strcmp(pName, "table:table-column")) {
        
        _closeColumn ();
        
    } else if (!UT_strcmp(pName, "table:table-row")) {
        
        _closeRow ();
        
    } else if (!UT_strcmp(pName, "table:table-cell")) {
        
        _closeCell ();
        
    } else if (!UT_strcmp(pName, "text:date") ||
        !UT_strcmp(pName, "text:time") ||
        !UT_strcmp(pName, "text:page-number") ||
        !UT_strcmp(pName, "text:page-count") ||
        !UT_strcmp(pName, "text:file-name") ||
        !UT_strcmp(pName, "text:paragraph-count") ||
        !UT_strcmp(pName, "text:word-count") ||
        !UT_strcmp(pName, "text:character-count") ||
        !UT_strcmp(pName, "text:initial-creator") ||
        !UT_strcmp(pName, "text:author-name") ||
        !UT_strcmp(pName, "text:description") ||
        !UT_strcmp(pName, "text:keywords") ||
        !UT_strcmp(pName, "text:subject") ||
        !UT_strcmp(pName, "text:title")) {
            
        m_bAcceptingText = true;
        
    } else if (!UT_strcmp(pName, "office:text")) {
        
        UT_ASSERT(m_bOnContentStream);
        
        // We were inside a <office:text> element.
        
        // We can now bring up the postponed parsing (headers and footers)
        rAction.bringUpPostponedElements(false);
        
    } else if (!UT_strcmp(pName, "style:header") ||
               !UT_strcmp(pName, "style:footer") ||
               !UT_strcmp(pName, "style:header-left") ||
               !UT_strcmp(pName, "style:footer-left")) {
                
        // We were inside a <style:header/footer> element.
        
        UT_ASSERT(!m_bOnContentStream);
        
        rAction.popState();
    }
    
    m_elementStack.endElement(pName);
}


/**
 * 
 */
void OD_TextContent_ListenerState::charData (
                            const XML_Char* pBuffer, int length)
{
    if (pBuffer && length && m_bAcceptingText && !m_bInTOC) {
        m_charData += UT_UCS4String (pBuffer, length, true);
    }
}


/**
 * 
 */
void OD_TextContent_ListenerState::_insertImage (const XML_Char** ppAtts)
{
    UT_Error error      = UT_OK;
    const XML_Char * width  = UT_getAttribute ("svg:width", ppAtts);
    const XML_Char * height = UT_getAttribute ("svg:height", ppAtts);
    const XML_Char * href   = UT_getAttribute ("xlink:href", ppAtts);

    m_imgCnt++;

    UT_ByteBuf img_buf;

    GsfInfile * pictures_dir =
        GSF_INFILE(gsf_infile_child_by_name(m_pGsfInfile, "Pictures"));


    // 9 == strlen("Pictures/");
    error = _loadStream(pictures_dir, href+9, img_buf);
    g_object_unref (G_OBJECT (pictures_dir));


    if (error != UT_OK) {
        return;
    }

    const char * mimetype   = UT_strdup ("image/png");
    IE_ImpGraphic * importer    = 0;
    FG_Graphic* pFG     = 0;
    UT_ByteBuf * pictData       = 0;

    UT_String propBuffer;
    UT_String propsName;

    error = IE_ImpGraphic::constructImporter (&img_buf, IEGFT_Unknown, &importer);

    if ((error != UT_OK) || !importer) {
        FREEP(mimetype);
        goto Cleanup;
    }

    error = importer->importGraphic(&img_buf, &pFG);
    if ((error != UT_OK) || !pFG) {
        // pictData is already freed in ~FG_Graphic
        FREEP(mimetype);
        goto Cleanup;
    }

    // TODO: can we get back a vector graphic?
    pictData = static_cast<FG_GraphicRaster *>(pFG)->getRaster_PNG();

    if (!pictData) {
        // i don't think that this could ever happen, but...
        FREEP(mimetype);
        error = UT_ERROR;
        goto Cleanup;
    }

    //
    // This next bit of code will set up our properties based on the image attributes
    //

    UT_String_sprintf(propBuffer, "width:%s; height:%s", width, height);
    UT_String_sprintf(propsName, "image%d", m_imgCnt);

    const XML_Char* propsArray[5];
    propsArray[0] = (XML_Char *)"props";
    propsArray[1] = (XML_Char *)propBuffer.c_str();
    propsArray[2] = (XML_Char *)"dataid";
    propsArray[3] = (XML_Char *)propsName.c_str();
    propsArray[4] = 0;

    if (!m_pAbiDocument->appendObject (PTO_Image, propsArray)) {
        FREEP(mimetype);
        goto Cleanup;
    }

    if (!m_pAbiDocument->createDataItem(propsName.c_str(), false,
    pictData, (void*)mimetype, NULL)) {
        goto Cleanup;
    }

    Cleanup:
    DELETEP(importer);
}


/**
 * 
 */
void OD_TextContent_ListenerState::_insertBookmark (const XML_Char* pName,
                                             const XML_Char* pType)
{
    const XML_Char* pPropsArray[5];
    pPropsArray[0] = (XML_Char *)"name";
    pPropsArray[1] = pName;
    pPropsArray[2] = (XML_Char *)"type";
    pPropsArray[3] = pType;
    pPropsArray[4] = 0;
    m_pAbiDocument->appendObject (PTO_Bookmark, pPropsArray);
}


/**
 * 
 */
void OD_TextContent_ListenerState::_flush ()
{
    if (m_charData.size () > 0 && m_bAcceptingText) {
        m_pAbiDocument->appendSpan (m_charData.ucs4_str(), m_charData.size ());
        m_charData.clear ();
    } 
}


/**
 * 
 */
void OD_TextContent_ListenerState::_openTable (const XML_Char **ppProps)
{
    m_pAbiDocument->appendStrux(PTX_SectionTable, NULL);
}


/**
 * 
 */
void OD_TextContent_ListenerState::_openColumn (const XML_Char** ppProps)
{
    m_col++;
}


/**
 * 
 */
void OD_TextContent_ListenerState::_openRow (const XML_Char** ppProps)
{
    m_row++;
    m_cel = 0;
}


/**
 * 
 */
void OD_TextContent_ListenerState::_openCell (const XML_Char** ppProps)
{
    UT_String attach;

    attach = UT_String_sprintf(
        "left-attach: %d; top-attach: %d; right-attach: %d; bot-attach: %d",
        m_cel, m_row-1, m_cel+1, m_row);

    m_cel++;

    const XML_Char *cell_props[3];
    cell_props[0] = "props";
    cell_props[1] = attach.c_str();
    cell_props[2] = 0;
    
    m_pAbiDocument->appendStrux(PTX_SectionCell,cell_props);
}


/**
 * 
 */
void OD_TextContent_ListenerState::_closeTable ()
{
    m_pAbiDocument->appendStrux(PTX_EndTable, NULL);

    m_row = m_cel = m_col = 0;
}


/**
 * 
 */
void OD_TextContent_ListenerState::_closeColumn ()
{
    m_col--;
}


/**
 * 
 */
void OD_TextContent_ListenerState::_closeRow ()
{
    m_col--;
}


/**
 * 
 */
void OD_TextContent_ListenerState::_closeCell ()
{
    m_pAbiDocument->appendStrux(PTX_EndCell,NULL);
}


/**
 * 
 */
bool OD_TextContent_ListenerState::_pushInlineFmt(const XML_Char ** atts)
{
    UT_uint32 start = m_vecInlineFmt.getItemCount()+1;
    UT_uint32 k;
    XML_Char* p;
    
    for (k=0; (atts[k]); k++)
    {
        if (!UT_XML_cloneString(p,atts[k])) {
            return false;
        }
        
        if (m_vecInlineFmt.addItem(p)!=0) {
            return false;
        }
        
        if (!m_stackFmtStartIndex.push((void*)start)) {
            return false;
        }
    }
        
    return true;
}


/**
 * 
 */
void OD_TextContent_ListenerState::_popInlineFmt(void)
{
    UT_uint32 start;
    
    if (!m_stackFmtStartIndex.pop((void **)&start))
        return;
        
    UT_uint32 k;
    UT_uint32 end = m_vecInlineFmt.getItemCount();
    const XML_Char* p;
    
    for (k=end; k>=start; k--) {
        
        p = (const XML_Char *)m_vecInlineFmt.getNthItem(k-1);
        m_vecInlineFmt.deleteNthItem(k-1);
        
        if (p)
            free((void *)p);
    }
}


/**
 * 
 */
UT_Error OD_TextContent_ListenerState::_loadStream ( GsfInfile * oo,
                 const char * stream,
                 UT_ByteBuf & buf )
{
    guint8 const *data = NULL;
    size_t len = 0;
    static const size_t BUF_SZ = 4096;
  
    buf.truncate (0);
    GsfInput * input = gsf_infile_child_by_name(oo, stream);

    if (!input)
        return UT_ERROR;
  
    if (gsf_input_size (input) > 0) {
        while ((len = gsf_input_remaining (input)) > 0) {
            len = UT_MIN (len, BUF_SZ);
            if (NULL == (data = gsf_input_read (input, len, NULL))) {
                g_object_unref (G_OBJECT (input));
                return UT_ERROR;
            }
            buf.append ((const UT_Byte *)data, len);
        }
    }
  
    g_object_unref (G_OBJECT (input));
    return UT_OK;
}


/**
 * Makes sure that an AbiWord section have already been created. Unlike
 * OpenDocument, AbiWord can't have paragraphs without a section to hold them.
 * 
 * @param pProps Properties for the AbiWord section (if a new one will be created).
 */
void OD_TextContent_ListenerState::_insureInSection(const XML_Char * pProps) 
{
    if (m_bInSection) {
      return;
    }
    
    const OD_Style_MasterPage* pMasterPageStyle;

    // For now we just use the Standard page master. AbiWord doesn't support
    // multiple page formats anyway.
    pMasterPageStyle = m_pStyles->getMasterPageStyle("Standard");
    UT_ASSERT(pMasterPageStyle);
    
    UT_ASSERT(pMasterPageStyle->getPageLayout());
    
    // The AbiWord section properties are taken part from the OpenDocument 
    // page layout (from the master page style) and part from the OpenDocument
    // section properties.
    
    // TODO: What happens if there are duplicated properties on the page layout
    // and on the section?

    UT_UTF8String allProps(pProps);
    allProps += pMasterPageStyle->getSectionProps();
    

    const XML_Char* atts[20];
    UT_uint8 i = 0;
    atts[i++] = "props";
    atts[i++] = allProps.utf8_str();
    
    if (!pMasterPageStyle->getAWEvenHeaderSectionID().empty()) {
        atts[i++] = "header-even";
        atts[i++] = pMasterPageStyle->getAWEvenHeaderSectionID().utf8_str();
    }
    
    if (!pMasterPageStyle->getAWHeaderSectionID().empty()) {
        atts[i++] = "header";
        atts[i++] = pMasterPageStyle->getAWHeaderSectionID().utf8_str();
    }
    
    if (!pMasterPageStyle->getAWEvenFooterSectionID().empty()) {
        atts[i++] = "footer-even";
        atts[i++] = pMasterPageStyle->getAWEvenFooterSectionID().utf8_str();
    }
    
    if (!pMasterPageStyle->getAWFooterSectionID().empty()) {
        atts[i++] = "footer";
        atts[i++] = pMasterPageStyle->getAWFooterSectionID().utf8_str();
    }
    
    atts[i] = 0; 
   
    m_pAbiDocument->appendStrux(PTX_Section, (const XML_Char**)atts);    

    m_bInSection = true;
    m_bAcceptingText = false;
}


/**
 * 
 */
void OD_TextContent_ListenerState::_insureInBlock(const XML_Char ** atts)
{
    if (m_bAcceptingText)
        return;

    _insureInSection(NULL);

    if (!m_bAcceptingText) {
        m_pAbiDocument->appendStrux(PTX_Block, (const XML_Char**)atts);    
        m_bAcceptingText = true;
    }
}
