/* AbiSource
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */
 
// Class definition include
#include "OD_StreamListener.h"

// Internal includes
#include "OD_SettingsStream_ListenerState.h"
#include "OD_StylesStream_ListenerState.h"
#include "OD_MetaStream_ListenerState.h"
#include "OD_ContentStream_ListenerState.h"
#include "OD_TextContent_ListenerState.h"
#include "OD_Postpone_ListenerState.h"
#include "OD_ListenerState.h"

// AbiWord includes
#include "ut_string.h"


/**
 * Constructor
 */
OD_StreamListener::OD_StreamListener(PD_Document* pAbiDocument,
                                     GsfInfile* pGsfInfile,
                                     OD_Office_Styles* pStyles)
                                     :
                                     m_pAbiDocument(pAbiDocument),
                                     m_pGsfInfile(pGsfInfile),
                                     m_pStyles(pStyles),
                                     m_pCurrentState(NULL),
                                     m_deleteCurrentWhenPop(false)
{
    UT_ASSERT(m_pAbiDocument);
    UT_ASSERT(m_pGsfInfile);
    UT_ASSERT(m_pStyles);
}


/**
 * Destructor
 */
OD_StreamListener::~OD_StreamListener()
{
    _clear();
    
    UT_ASSERT(m_postponedParsing.getItemCount() == 0);
    UT_VECTOR_PURGEALL(OD_Postpone_ListenerState*, m_postponedParsing);
}

 
/**
 * 
 */
void OD_StreamListener::startElement (const XML_Char* pName,
                                      const XML_Char** ppAtts)
{

    m_stateAction.reset();
    m_pCurrentState->startElement(pName, ppAtts, m_stateAction);
    
    if (m_stateAction.getAction() != m_stateAction.ACTION_NONE) {
        _handleStateAction();
        m_pCurrentState->startElement(pName, ppAtts, m_stateAction);
    }
    
}


/**
 * 
 */
void OD_StreamListener::endElement (const XML_Char* pName)
{
    
    m_stateAction.reset();
    m_pCurrentState->endElement(pName, m_stateAction);
    
    if (m_stateAction.getAction() != m_stateAction.ACTION_NONE) {
        _handleStateAction();
        m_pCurrentState->endElement(pName, m_stateAction);
    }
}


/**
 * 
 */
void OD_StreamListener::charData (const XML_Char* pBuffer, int length)
{
    
    m_pCurrentState->charData(pBuffer, length);
}


/**
 * Sets the current state of the stream listener.
 * 
 * @param pStateName The name of the state
 * @return An error if the state name is not recognized.
 */
UT_Error OD_StreamListener::setState(const char* pStateName)
{
    
    _clear();
    
    m_pCurrentState = _createState(pStateName);
    
    if (m_pCurrentState) {
        return UT_OK;
    } else {
        return UT_ERROR;
    }
}


/**
 * Push or pop the stack according to the action stated by the current state.
 */
void OD_StreamListener::_handleStateAction ()
{
    OD_StreamListener::StackCell stackCell;
    
    switch (m_stateAction.getAction()) {
        
        case OD_ListenerStateAction::ACTION_PUSH:
        
            m_stateStack.push_back(
                OD_StreamListener::StackCell(m_pCurrentState, m_deleteCurrentWhenPop));
                
            if (m_stateAction.getState() != NULL) {
                m_pCurrentState = m_stateAction.getState();
                m_deleteCurrentWhenPop = m_stateAction.getDeleteWhenPop();
            } else {
                m_pCurrentState = _createState(
                    m_stateAction.getStateName().c_str());
                    
                m_deleteCurrentWhenPop = true;
            }
            
            UT_ASSERT(m_pCurrentState);
            
            break;
            
            
        case OD_ListenerStateAction::ACTION_POP:
            
            stackCell = m_stateStack.getLastItem();
            
            if (m_deleteCurrentWhenPop) {
                DELETEP(m_pCurrentState);
            }
            
            m_pCurrentState = stackCell.m_pState;
            m_deleteCurrentWhenPop = stackCell.m_deleteWhenPop;
            
            m_stateStack.pop_back();
            break;


        case OD_ListenerStateAction::ACTION_POSTPONE:
            // If the state wants to come back later he shouldn't be deleted.
            UT_ASSERT(!m_deleteCurrentWhenPop);
            
            OD_Postpone_ListenerState* pPostponeState;
            
            pPostponeState = new OD_Postpone_ListenerState(m_pCurrentState);
            m_postponedParsing.addItem(pPostponeState);
            
            m_stateStack.push_back(
                OD_StreamListener::StackCell(m_pCurrentState, m_deleteCurrentWhenPop));
                
            m_pCurrentState = pPostponeState;
            m_deleteCurrentWhenPop = false;
            
            UT_ASSERT(m_pCurrentState);
            
            break;
            
        case OD_ListenerStateAction::ACTION_BRINGUP:
            UT_uint32 i;
            
            for (i=0; i<m_postponedParsing.getItemCount(); i++) {
                _resumeParsing(m_postponedParsing[i]);
            }
            
            UT_VECTOR_PURGEALL(OD_Postpone_ListenerState*, m_postponedParsing);
            m_postponedParsing.clear();
            
            if (!m_stateAction.getComeBackAfter()) {
                m_stateAction.popState();
                this->_handleStateAction();
            }
            break;
    };
}


/**
 * Clear the state stack.
 */
void OD_StreamListener::_clear ()
{
    
    if (m_pCurrentState && m_deleteCurrentWhenPop) {
        DELETEP(m_pCurrentState);
    } else {
        m_pCurrentState = NULL;
    }
    
    UT_uint32 i;
    OD_StreamListener::StackCell cell;
    for (i=0; i < m_stateStack.getItemCount(); i++) {
        cell = m_stateStack.getNthItem(i);
        if (cell.m_deleteWhenPop) {
            DELETEP(cell.m_pState);
        }
    }
    m_stateStack.clear();
}


/**
 * Create a state given its name.
 * 
 * @param pStateName Tha name of the state to be created.
 */
OD_ListenerState* OD_StreamListener::_createState(const char* pStateName) {
    
    OD_ListenerState* pState = NULL;
    
    if (!UT_strcmp("StylesStream", pStateName)) {
        
        pState = new OD_StylesStream_ListenerState(m_pAbiDocument, m_pGsfInfile,
                                                   m_pStyles);
        
    } else if (!UT_strcmp("MetaStream", pStateName)) {
        
        pState = new OD_MetaStream_ListenerState(m_pAbiDocument);
        
    } else if (!UT_strcmp("SettingsStream", pStateName)) {
        
        pState = new OD_SettingsStream_ListenerState();
        
    } else if (!UT_strcmp("ContentStream", pStateName)) {
        
        pState = new OD_ContentStream_ListenerState(m_pAbiDocument, m_pGsfInfile,
                                                    m_pStyles);

    } else if (!UT_strcmp("TextContent", pStateName)) {
        
        pState = new OD_TextContent_ListenerState(m_pAbiDocument, m_pGsfInfile,
                                                  m_pStyles);
    }
    
    return pState;
}


/**
 * Resumes the parsing of a XML element that was postponed.
 */
void OD_StreamListener::_resumeParsing(OD_Postpone_ListenerState* pPostponeState){
    UT_uint32 i, count;
    OD_ListenerState* pOriginalState = m_pCurrentState;
    OD_Postpone_ListenerState::StartElementCall* pStartCall = NULL;
    OD_Postpone_ListenerState::EndElementCall* pEndCall = NULL;
    OD_Postpone_ListenerState::CharDataCall* pCharDataCall = NULL;
    
    m_stateStack.push_back(
        OD_StreamListener::StackCell(m_pCurrentState, m_deleteCurrentWhenPop));
    
    m_pCurrentState = pPostponeState->getParserState();
    m_deleteCurrentWhenPop = false;
    
    count = pPostponeState->m_XMLCalls.getItemCount();
    for (i=0; i<count; i++) {
        switch ( (pPostponeState->m_XMLCalls[i])->getType() ) {
            
            case OD_Postpone_ListenerState::XMLCallType_StartElement:
                pStartCall = (OD_Postpone_ListenerState::StartElementCall*)
                                pPostponeState->m_XMLCalls[i];
                this->startElement(pStartCall->m_pName,
                                   (const XML_Char**) pStartCall->m_ppAtts);
                break;
                
                
            case OD_Postpone_ListenerState::XMLCallType_EndElement:
                pEndCall = (OD_Postpone_ListenerState::EndElementCall*)
                                pPostponeState->m_XMLCalls[i];
                this->endElement(pEndCall->m_pName);
                break;
                
            case OD_Postpone_ListenerState::XMLCallType_CharData:
                pCharDataCall = (OD_Postpone_ListenerState::CharDataCall*)
                                pPostponeState->m_XMLCalls[i];
                this->charData(pCharDataCall->m_pBuffer, pCharDataCall->m_length);
                break;
        }
    }
    
    
    // Return to the original state (if not there already).
    if (pOriginalState != m_pCurrentState) {
        OD_StreamListener::StackCell stackCell;
            
        stackCell = m_stateStack.getLastItem();
        
        if (m_deleteCurrentWhenPop) {
            DELETEP(m_pCurrentState);
        }
        
        m_pCurrentState = stackCell.m_pState;
        m_deleteCurrentWhenPop = stackCell.m_deleteWhenPop;
        
        m_stateStack.pop_back();
        
        UT_ASSERT(pOriginalState == m_pCurrentState);
    }
}
