/* AbiSource
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */

#ifndef _OD_OFFICE_STYLES_H_
#define _OD_OFFICE_STYLES_H_

// AbiWord includes
#include <ut_hash.h>
#include <ut_vector.h>

// Internal classes
class OD_Style_Style;
class OD_Style_PageLayout;
class OD_Style_MasterPage;

// AbiWord classes
class PD_Document;



/**
 * This class holds all the styles defined by the OpenDocument stream, both the
 * common (defined inside <office:styles>), automatic (definded
 * inside <office:automatic-styles>) and master page ones (defined inside
 * <office:master-styles>).
 * 
 * It includes the page layouts (<style:page-layout>).
 * 
 * Usage:
 * 1 - Add (and parse) all styles
 * 2 - Call addedAllStyles
 */
class OD_Office_Styles {
public:

    OD_Office_Styles() : m_pParagraphDefaultStyle(NULL) {}
    ~OD_Office_Styles();

    OD_Style_Style* addStyle(const XML_Char** ppAtts, bool bAutomatic,
                             bool bOnContentStream);

    OD_Style_PageLayout* addPageLayout(const XML_Char** ppAtts);

    OD_Style_MasterPage* addMasterPage(const XML_Char** ppAtts,
                                       PD_Document* pDocument);

    OD_Style_Style* addDefaultStyle(const XML_Char** ppAtts);

    /**
     * Will do some post-processing and then define all AbiWord styles.
     */
    inline void addedAllStyles(PD_Document* pDocument) {
        _fixStyles();
        _linkStyles();
        _buildAbiPropsAttrString();
        _defineAbiStyles(pDocument);
    }
    
    const OD_Style_Style* getTextStyle(const XML_Char* pStyleName,
                                       bool bOnContentStream) const;
    
    const OD_Style_Style* getParagraphStyle(const XML_Char* pStyleName,
                                            bool bOnContentStream) const;
    
    const OD_Style_Style* getSectionStyle(const XML_Char* pStyleName,
                                          bool bOnContentStream) const;
    
    inline const OD_Style_Style* getDefaultParagraphStyle() const {
        return m_pParagraphDefaultStyle;
    }
    
    inline const OD_Style_PageLayout* getPageLayoutStyle(
                                             const XML_Char* pStyleName) const {
        return m_pageLayoutStyles.pick(pStyleName);
    }
    
    inline const OD_Style_MasterPage* getMasterPageStyle(
                                             const XML_Char* pStyleName) const {
        return m_masterPageStyles.pick(pStyleName);
    }
    
private:

    void _buildAbiPropsAttrString();
    void _defineAbiStyles(PD_Document* pDocument) const;
    
    void _linkStyles();
    
    void _linkStyles(UT_GenericStringMap<OD_Style_Style*>& rStyles,
                     const OD_Style_Style* pDefaultStyle = NULL);
                     
    void _fixStyles();
    void _removeParagraphStyleStyle(const UT_UTF8String& rName);

    ////
    // The styles are separated per family.

    // <style:style style:family="text"> 
    UT_GenericStringMap<OD_Style_Style*> m_textStyleStyles;

    // <style:style style:family="paragraph"> 
    UT_GenericStringMap<OD_Style_Style*> m_paragraphStyleStyles;

    // <style:style style:family="section"> 
    UT_GenericStringMap<OD_Style_Style*> m_sectionStyleStyles;
    
    ////
    // Styles defined inside the content stream, only automatic styles.
    // (<office:document-content> element)
    
    // <style:style style:family="text"> 
    UT_GenericStringMap<OD_Style_Style*> m_textStyleStyles_contentStream;

    // <style:style style:family="paragraph"> 
    UT_GenericStringMap<OD_Style_Style*> m_paragraphStyleStyles_contentStream;

    // <style:style style:family="section"> 
    UT_GenericStringMap<OD_Style_Style*> m_sectionStyleStyles_contentStream;
    
    
    
    // <style:page-layout>
    UT_GenericStringMap<OD_Style_PageLayout*> m_pageLayoutStyles;
    
    // <style:master-page>
    UT_GenericStringMap<OD_Style_MasterPage*> m_masterPageStyles;

    // <style:default-style style:family="paragraph">
    // Maps to the "Normal" style in AbiWord/
    OD_Style_Style* m_pParagraphDefaultStyle;
    
    
    // The paragraph styles that get removed due to lack of properties have
    // their names stored here.
    //
    // When a style whose name lies on that list is requested, we send back
    // the default paragraph style ("Normal", in AbiWord).
    UT_GenericVector<UT_UTF8String*> m_removedParagraphStyles;
};

#endif //_OD_OFFICE_STYLES_H_
