/* AbiSource
 * 
 * Copyright (C) 2005 Daniel d'Andrada T. de Carvalho
 * <daniel.carvalho@indt.org.br>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */
 
// Class definition include
#include "OD_Office_Styles.h"

// Internal inlcudes
#include "OD_Style_Style.h"
#include "OD_Style_PageLayout.h"
#include "OD_Style_MasterPage.h"
 
// AbiWord includes
#include <ut_misc.h>
#include <pd_Document.h>
#include <ut_debugmsg.h>





/**
 * Destructor
 */
OD_Office_Styles::~OD_Office_Styles() {
    
    UT_GenericVector<OD_Style_Style*>* pStyleVector;
    UT_GenericVector<OD_Style_PageLayout*>* pPageLayoutVector;
    UT_GenericVector<OD_Style_MasterPage*>* pMasterPageVector;
    UT_uint32 i, count;
    
    
    UT_VECTOR_PURGEALL(UT_UTF8String*, m_removedParagraphStyles);
    
    pStyleVector = m_textStyleStyles.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }
    

    pStyleVector = m_paragraphStyleStyles.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }


    pStyleVector = m_sectionStyleStyles.enumerate();
    count = pStyleVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pStyleVector)[i];
    }
    
    
    pPageLayoutVector = m_pageLayoutStyles.enumerate();
    count = pPageLayoutVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pPageLayoutVector)[i];
    }
    
    
    pMasterPageVector = m_masterPageStyles.enumerate();
    count = pMasterPageVector->getItemCount();
    for (i=0; i<count; i++) {
        delete (*pMasterPageVector)[i];
    }

    DELETEP(m_pParagraphDefaultStyle);
}
 




/**
 * Adds a <style:style> (OD_Style_Style class).
 * 
 * @param bAutomatic true if the style is an OpenDocument "automatic style".
 * 
 * @return The address of the newly created OD_Style_Style or NULL, if the
 *         specified style is not currently supported (like graphic styles).
 */
OD_Style_Style* OD_Office_Styles::addStyle(const XML_Char** ppAtts,
                                           bool bAutomatic, bool bOnContentStream) {

    const XML_Char* pAttrValue;
    OD_Style_Style* pStyle;
    bool ok;
    
    pAttrValue = UT_getAttribute("style:family", ppAtts);
    UT_ASSERT(pAttrValue);
    
    if (bOnContentStream) {
    
        if(!UT_strcmp(pAttrValue, "text")) {
            
            pStyle = new OD_Style_Style(bAutomatic);
            pAttrValue = UT_getAttribute("style:name", ppAtts);
            ok = m_textStyleStyles_contentStream.insert(pAttrValue, pStyle);
            
        } else if(!UT_strcmp(pAttrValue, "paragraph")) {
            
            pStyle = new OD_Style_Style(bAutomatic);
            pAttrValue = UT_getAttribute("style:name", ppAtts);
            ok = m_paragraphStyleStyles_contentStream.insert(pAttrValue, pStyle);
            
        } else if(!UT_strcmp(pAttrValue, "section")) {
            
            pStyle = new OD_Style_Style(bAutomatic);
            pAttrValue = UT_getAttribute("style:name", ppAtts);
            ok = m_sectionStyleStyles_contentStream.insert(pAttrValue, pStyle);
            
        } else {
            // We don't recognize it, yet.
            return NULL;
        }
    
    } else {
        
        if(!UT_strcmp(pAttrValue, "text")) {
            
            pStyle = new OD_Style_Style(bAutomatic);
            pAttrValue = UT_getAttribute("style:name", ppAtts);
            ok = m_textStyleStyles.insert(pAttrValue, pStyle);
            
        } else if(!UT_strcmp(pAttrValue, "paragraph")) {
            
            pStyle = new OD_Style_Style(bAutomatic);
            pAttrValue = UT_getAttribute("style:name", ppAtts);
            ok = m_paragraphStyleStyles.insert(pAttrValue, pStyle);
            
        } else if(!UT_strcmp(pAttrValue, "section")) {
            
            pStyle = new OD_Style_Style(bAutomatic);
            pAttrValue = UT_getAttribute("style:name", ppAtts);
            ok = m_sectionStyleStyles.insert(pAttrValue, pStyle);
            
        } else {
            // We don't recognize it, yet.
            return NULL;
        }
        
    }
    
    UT_ASSERT(ok);
    
    return pStyle;   
}





/**
 * Adds a <style:page-layout> (OD_Style_PageLayout class)
 * 
 * @return The address of the newly created OD_Style_PageLayout.
 */
OD_Style_PageLayout* OD_Office_Styles::addPageLayout(const XML_Char** ppAtts) {
                               
    const XML_Char* pAttrValue;
    OD_Style_PageLayout* pStyle;
    bool ok;
                             
    pStyle = new OD_Style_PageLayout();
    pAttrValue = UT_getAttribute("style:name", ppAtts);
    ok = m_pageLayoutStyles.insert(pAttrValue, pStyle);
    
    UT_ASSERT(ok);
    
    return pStyle;
}





/**
 * Adds a <style:master-page> (OD_Style_MasterPage class)
 * 
 * @return The address of the newly created OD_Style_MasterPage.
 */
OD_Style_MasterPage* OD_Office_Styles::addMasterPage(const XML_Char** ppAtts,
                                                     PD_Document* pDocument) {
                                                        
    const XML_Char* pAttrValue;
    OD_Style_MasterPage* pStyle;
    bool ok;
                             
    pStyle = new OD_Style_MasterPage(pDocument);
    pAttrValue = UT_getAttribute("style:name", ppAtts);
    ok = m_masterPageStyles.insert(pAttrValue, pStyle);
    
    UT_ASSERT(ok);
    
    return pStyle;
}





/**
 * 
 */
OD_Style_Style* OD_Office_Styles::addDefaultStyle(const XML_Char** ppAtts) {
    
    const XML_Char* pAttr;
    
    pAttr = UT_getAttribute("style:family", ppAtts);
    UT_ASSERT(pAttr);

    if (!UT_strcmp("paragraph", pAttr)) {
        m_pParagraphDefaultStyle = new OD_Style_Style(false);
        return m_pParagraphDefaultStyle;
    } else {
        // Not currently supported
        return NULL;
    }
}





/**
 * Links every style with its parent and next ones.
 */
void OD_Office_Styles::_linkStyles() {
    
    _linkStyles(m_textStyleStyles);
    _linkStyles(m_paragraphStyleStyles, m_pParagraphDefaultStyle);
    _linkStyles(m_sectionStyleStyles);
    
    
    
    // Link the master styles to their page layouts
    
    UT_GenericVector<OD_Style_MasterPage*>* pStylesVec;
    UT_uint32 i, count;
    OD_Style_MasterPage* pStyle;
    OD_Style_PageLayout* pLayout;
    
    pStylesVec = m_masterPageStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        pStyle = (*pStylesVec)[i];
        
        pLayout = m_pageLayoutStyles.pick(pStyle->getLayoutName().utf8_str());
        UT_ASSERT(pLayout);
        
        pStyle->setLayoutStylePointer(pLayout);
    }
}





/**
 * Helper funtion for linkStyles()
 */   
void OD_Office_Styles::_linkStyles(
                        UT_GenericStringMap<OD_Style_Style*>& rStyles,
                        const OD_Style_Style* pDefaultStyle) {
    
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    UT_uint32 i, count;
    OD_Style_Style* pStyle;
    const OD_Style_Style* pOtherStyle;
    
    pStylesVec = rStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        pStyle = (*pStylesVec)[i];
        
        if (!pStyle->getParentStyleName().empty()) {
            
            pOtherStyle = rStyles.pick(
                pStyle->getParentStyleName().utf8_str());
                
            if (pOtherStyle) {
                pStyle->setParentStylePointer(pOtherStyle);
                
            } else if (pDefaultStyle) {
                
                if (pDefaultStyle->getName() == pStyle->getParentStyleName()) {
                    pStyle->setParentStylePointer(pDefaultStyle);
                } else {
                    UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
                }
                
            } else {
                UT_ASSERT(UT_SHOULD_NOT_HAPPEN);
            }
                
        }
        
        if (!pStyle->getNextStyleName().empty()) {
            
            pOtherStyle = rStyles.pick(
                pStyle->getNextStyleName().utf8_str());
            UT_ASSERT(pOtherStyle);
            
            pStyle->setNextStylePointer(pOtherStyle);
        }
    }
}





/**
 * 
 */
void OD_Office_Styles::_defineAbiStyles(PD_Document* pDocument) const {
    
    UT_uint32 i, count;
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    
    m_pParagraphDefaultStyle->defineAbiStyle(pDocument);
    
    pStylesVec = m_textStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->defineAbiStyle(pDocument);
    }
    
    
    pStylesVec = m_paragraphStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->defineAbiStyle(pDocument);
    }
    
    // AbiWord doesn't have section styles.
    




    // I will just use the first master page style.
    UT_GenericVector<OD_Style_MasterPage*>* pMasterStylesVec;
    
    pMasterStylesVec = m_masterPageStyles.enumerate();
    
    (*pMasterStylesVec)[0]->definePageSizeTag(pDocument);
}





/**
 * 
 */
void OD_Office_Styles::_buildAbiPropsAttrString() {
    
    UT_uint32 i, count;
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    
    UT_ASSERT(m_pParagraphDefaultStyle);
    m_pParagraphDefaultStyle->buildAbiPropsAttrString();
    
    
    pStylesVec = m_textStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
    
    
    pStylesVec = m_paragraphStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
    
    
    pStylesVec = m_sectionStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        (*pStylesVec)[i]->buildAbiPropsAttrString();
    }
}





/**
 * Fix any problems encountered on the added styles
 */
void OD_Office_Styles::_fixStyles() {
    // Problem 1: We can't have styles without properties
    //
    // The "Standard" paragraph style usually comes empty
    // (I have never seen otherwise)
    
    UT_uint32 i, count;
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    OD_Style_Style* pStyle;
    UT_UTF8String styleName;
    bool noneFound;
    
    do {
        pStylesVec = m_paragraphStyleStyles.enumerate();
        UT_ASSERT(pStylesVec);
        
        noneFound = true;
        count = pStylesVec->getItemCount();
        for (i=0; i<count; i++) {
            if ( !((*pStylesVec)[i]->hasProperties()) ) {
                styleName = (*pStylesVec)[i]->getName();
                i=count;
                noneFound = false;
            }
        }
        
        if (!noneFound) {
            _removeParagraphStyleStyle(styleName);
        }
    } while (!noneFound);
}





/**
 * 
 */
void OD_Office_Styles::_removeParagraphStyleStyle(const UT_UTF8String& rName) {
    
    UT_uint32 i, count;
    UT_GenericVector<OD_Style_Style*>* pStylesVec;
    OD_Style_Style* pStyle;
    UT_UTF8String styleName;
    
    // Remove the style itself
    m_paragraphStyleStyles.remove(rName.utf8_str(), NULL);
    m_removedParagraphStyles.addItem(new UT_UTF8String(rName.utf8_str()));
    
    
    // Fix all references to it
    pStylesVec = m_paragraphStyleStyles.enumerate();
    UT_ASSERT(pStylesVec);
    
    count = pStylesVec->getItemCount();
    for (i=0; i<count; i++) {
        if ((*pStylesVec)[i]->getParentName() == rName) {
            (*pStylesVec)[i]->setParentName("Normal");
        }
        
        if ((*pStylesVec)[i]->getNextStyleName() == rName) {
            (*pStylesVec)[i]->setNextStyleName("Normal");
        }
    }

}





/**
 * Returns the specified paragraph style
 * 
 * @param pStyleName The name of the style wanted.
 */
const OD_Style_Style* OD_Office_Styles::getParagraphStyle(
                                             const XML_Char* pStyleName,
                                             bool bOnContentStream) const {
                                                
    OD_Style_Style* pStyle;
    
    if (bOnContentStream) {
        pStyle = m_paragraphStyleStyles_contentStream.pick(pStyleName);
        if (!pStyle) {
            // Should be a regular style (not automatic).
            pStyle = m_paragraphStyleStyles.pick(pStyleName);
        }
    } else {
        pStyle = m_paragraphStyleStyles.pick(pStyleName);
    }
    
    if (!pStyle) {
        // We haven't found it. Have we removed it (on _fixStyles())?
        UT_UTF8String name (pStyleName);
        UT_uint32 i, count;
        
        count = m_removedParagraphStyles.getItemCount();
        for (i=0; i<count; i++) {
            if (name == *(m_removedParagraphStyles[i])) {
                return m_pParagraphDefaultStyle;
            }
        }
        
        // This style never existed.
        return NULL;
    } else {
        return pStyle;
    }
}


/**
 * 
 */
const OD_Style_Style* OD_Office_Styles::getTextStyle(const XML_Char* pStyleName,
                                              bool bOnContentStream) const {
    OD_Style_Style* pStyle;
    
    if (bOnContentStream) {
        pStyle = m_textStyleStyles_contentStream.pick(pStyleName);
        
        if (!pStyle) {
            // Should be a regular style (not automatic).
            pStyle = m_textStyleStyles.pick(pStyleName);
        }
        
        return pStyle;
    } else {
        return m_textStyleStyles.pick(pStyleName);
    }
}
    

/**
 * 
 */ 
const OD_Style_Style* OD_Office_Styles::getSectionStyle(const XML_Char* pStyleName,
                                                 bool bOnContentStream) const {
    OD_Style_Style* pStyle;
    
    if (bOnContentStream) {
        pStyle = m_sectionStyleStyles_contentStream.pick(pStyleName);
        
        if (!pStyle) {
            // Should be a regular style (not automatic).
            pStyle = m_sectionStyleStyles.pick(pStyleName);
        }
        
        return pStyle;
    } else {
        return m_sectionStyleStyles.pick(pStyleName);
    }
}