/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiWord / AbiBits / HanAbi
 *
 * Copyright (C) 2003 AbiSource, Inc.
 * Copyright (C) 2003 Francis James Franklin <fjf@alinameridon.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <unistd.h>
#include <fcntl.h>

#include "ut_exception.h"
#include "ut_debugmsg.h"
#include "ut_Tree.h"

#import "HanAbiOptions.h"
#import "HanAbiPreferences.h"

static const char * s_xml_dtd_declaration = "<?xml version=\"1.0\"?>\n"
"<!DOCTYPE preferences PUBLIC \"-//ABISOURCE//DTD HANABI 1.0 Preferences//EN\"\n"
"              \"http://www.abisource.com/2003/HanAbi/preferences/HanAbi.mod\">\n";

@implementation HanAbiPreferences

static HanAbiPreferences * s_systemPrefs = nil;
static HanAbiPreferences * s_userPrefs   = nil;

+ (HanAbiPreferences *)systemPrefs
{
	if (s_systemPrefs == nil)
		{
			s_systemPrefs = [[HanAbiPreferences alloc] init];
			if (s_systemPrefs)
				if ([s_systemPrefs loadSystemPrefs] == NO)
					{
						[s_systemPrefs release];
						s_systemPrefs = nil;
					}
		}
	return s_systemPrefs;
}

+ (HanAbiPreferences *)userPrefs
{
	if (s_userPrefs == nil)
		{
			s_userPrefs = [[HanAbiPreferences alloc] init];
			if (s_userPrefs)
				if ([s_userPrefs loadUserPrefs] == NO)
					{
						[s_userPrefs release];
						s_userPrefs = nil;
					}
		}
	return s_userPrefs;
}

- (HanAbiPreferences *)pluginPrefs:(NSString *)plugin
{
	if (m_parent)
		return [m_parent pluginPrefs:plugin];

	if (!m_tree || !plugin) // erk!
		return nil;
	if ([plugin length] == 0) // erk!
		return nil;

	HanAbiPreferences * prefs = nil;
	prefs = (HanAbiPreferences *) [[HanAbiPreferences alloc] initWithParent:self withTree:m_tree forPlugin:plugin];

	if (prefs)
		[prefs autorelease];

	return prefs;
}

- (HanAbiPreferences *)toolbarPrefs:(NSString *)toolbar
{
	if (m_parent)
		return [m_parent toolbarPrefs:toolbar];

	if (!m_tree || !toolbar) // erk!
		return nil;
	if ([toolbar length] == 0) // erk!
		return nil;

	HanAbiPreferences * prefs = nil;
	prefs = (HanAbiPreferences *) [[HanAbiPreferences alloc] initWithParent:self withTree:m_tree forToolbar:toolbar];

	if (prefs)
		[prefs autorelease];

	return prefs;
}

- (id)initWithParent:(HanAbiPreferences *)prefs withTree:(UT_Tree *)tree forPlugin:(NSString *)pluginName
{
	[super init];

	m_parent = prefs;
	m_tree = tree;

	m_prefs = 0;

	if (!prefs || !tree || !pluginName) // erk!
		{
			[self release];
			return nil;
		}
	if ([pluginName length] == 0) // erk!
		{
			[self release];
			return nil;
		}
	const UT_Element * preferences = tree->root ();
	const UT_Element * plugins = 0;
	const UT_Element * plugin  = 0;
	const UT_Element * options = 0;

	if (preferences)
		{
			UT_uint32 children = preferences->children ();

			for (UT_uint32 i = 0; i < children; i++)
				{
					const UT_Element * child = preferences->child (i);

					if (child->ElementTag () == "plugins")
						{
							plugins = child;
							break;
						}
				}
		}
	if (plugins)
		{
			UT_uint32 children = preferences->children ();

			for (UT_uint32 i = 0; i < children; i++)
				{
					const UT_Element * child = preferences->child (i);

					if (child->ElementTag () == "plugin")
						{
							const UT_UTF8String * name = child->Attr ("name");
							if (name)
								if (*name == [pluginName UTF8String])
									{
										plugin = child;
										break;
									}
						}
				}
		}
	if (plugins && !plugin)
		{
			bool okay = true;

			UT_Element * element = 0;

			UT_TRY
				{
					element = new UT_Element("plugin","");
				}
			UT_CATCH(...)
				{
					element = 0;
				}
			if (element == 0)
				okay = false;

			if (okay)
				okay = element->insAttr ("name", [pluginName UTF8String]);
			if (okay)
				okay = const_cast<UT_Element *>(plugins)->appendNode (element);
			else if (element)
				delete element;

			if (okay)
				plugin = element;
		}
	if (plugin)
		{
			bool okay = true;

			UT_uint32 children = plugin->children ();

			for (UT_uint32 i = 0; i < children; i++)
				{
					const UT_Element * child = plugin->child (i);

					if (child->ElementTag () == "options")
						{
							options = child;
							break;
						}
				}
			if (!options)
				{
					UT_Element * element = 0;

					UT_TRY
						{
							element = new UT_Element("options","");
						}
					UT_CATCH(...)
						{
							element = 0;
						}
					if (element == 0)
						okay = false;

					if (okay)
						okay = const_cast<UT_Element *>(plugin)->insertNode (element, 0);
					if (okay)
						options = element;
				}
		}
	if (!options)
		{
			[self release];
			return nil;
		}

	m_prefs = plugin;

	return self;
}

- (id)initWithParent:(HanAbiPreferences *)prefs withTree:(UT_Tree *)tree forToolbar:(NSString *)toolbarName
{
	[super init];

	m_parent = prefs;
	m_tree = tree;

	m_prefs = 0;

	if (!prefs || !tree || !toolbarName) // erk!
		{
			[self release];
			return nil;
		}
	if ([toolbarName length] == 0) // erk!
		{
			[self release];
			return nil;
		}
	const UT_Element * preferences = tree->root ();
	const UT_Element * toolbars = 0;
	const UT_Element * toolbar  = 0;
	const UT_Element * options  = 0;

	if (preferences)
		{
			UT_uint32 children = preferences->children ();

			for (UT_uint32 i = 0; i < children; i++)
				{
					const UT_Element * child = preferences->child (i);

					if (child->ElementTag () == "toolbars")
						{
							toolbars = child;
							break;
						}
				}
		}
	if (toolbars)
		{
			UT_uint32 children = preferences->children ();

			for (UT_uint32 i = 0; i < children; i++)
				{
					const UT_Element * child = preferences->child (i);

					if (child->ElementTag () == "toolbar")
						{
							const UT_UTF8String * name = child->Attr ("name");
							if (name)
								if (*name == [toolbarName UTF8String])
									{
										toolbar = child;
										break;
									}
						}
				}
		}
	if (toolbars && !toolbar)
		{
			bool okay = true;

			UT_Element * element = 0;

			UT_TRY
				{
					element = new UT_Element("toolbar","");
				}
			UT_CATCH(...)
				{
					element = 0;
				}
			if (element == 0)
				okay = false;

			if (okay)
				okay = element->insAttr ("name", [toolbarName UTF8String]);
			if (okay)
				okay = const_cast<UT_Element *>(toolbars)->appendNode (element);
			else if (element)
				delete element;

			if (okay)
				toolbar = element;
		}
	if (toolbar)
		{
			bool okay = true;

			UT_uint32 children = toolbar->children ();

			for (UT_uint32 i = 0; i < children; i++)
				{
					const UT_Element * child = toolbar->child (i);

					if (child->ElementTag () == "options")
						{
							options = child;
							break;
						}
				}
			if (!options)
				{
					UT_Element * element = 0;

					UT_TRY
						{
							element = new UT_Element("options","");
						}
					UT_CATCH(...)
						{
							element = 0;
						}
					if (element == 0)
						okay = false;

					if (okay)
						okay = const_cast<UT_Element *>(toolbar)->insertNode (element, 0);
					if (okay)
						options = element;
				}
		}
	if (!options)
		{
			[self release];
			return nil;
		}

	m_prefs = toolbar;

	return self;
}

- (id)init
{
	[super init];

	m_parent = nil;

	m_tree = 0;
	m_prefs = 0;

	m_changed = NO;
	m_system = NO;

	return self;
}

- (void)dealloc
{
	if (m_parent == nil)
		if (m_tree)
			delete m_tree;

	[super dealloc];
}

- (NSString *)valueForOption:(NSString *)key
{
	if (!m_prefs)
		return nil;

	const char * value = HanAbi_GetOption (m_prefs, [key UTF8String]);

	return value ? [NSString stringWithUTF8String:value] : nil;
}

- (BOOL)setValue:(NSString *)value forOption:(NSString *)key
{
	if (m_prefs == 0)
		return NO;

	BOOL okay = NO;

	if ([self prefsCanChange])
		{
			okay = HanAbi_SetOption (m_prefs, [key UTF8String], [value UTF8String]) ? YES : NO;

			if (okay)
				[self setPrefsHaveChanged];
		}
	return okay;
}

- (BOOL)prefsCanChange
{
	if (m_parent)
		return [m_parent prefsCanChange];

	return (m_tree && m_prefs && (m_system == NO)) ? YES : NO;
}

- (BOOL)prefsHaveChanged
{
	if (m_parent)
		return [m_parent prefsHaveChanged];

	return m_changed;
}

- (void)setPrefsHaveChanged
{
	if (m_parent)
		[m_parent setPrefsHaveChanged];
	else	
		m_changed = YES;
}

- (BOOL)loadPrefsFromFile:(NSString *)filename generateIfNone:(BOOL)generate
{
	NSFileManager * FM = [NSFileManager defaultManager];

	BOOL generatePrefs = NO;
	BOOL isDirectory = NO;

	if ([FM fileExistsAtPath:filename isDirectory:&isDirectory] == NO)
		{
			if (!generate)
				return NO;
			generatePrefs = YES;
		}
	else if (isDirectory == YES)
		return NO;

	NSData * data = nil;

	if (!generatePrefs)
		{
			data = (NSData *) [NSData dataWithContentsOfFile:filename];

			if ([data length] == 0)
				return NO;
		}

	UT_TRY
		{
			m_tree = new UT_Tree;
		}
	UT_CATCH(...)
		{
			m_tree = 0;
		}
	if (m_tree == 0)
		return NO;

	bool okay = true;

	if (generatePrefs)
		{
			UT_Default_Node * default_node = 0;

			UT_TRY
				{
					default_node = new UT_Default_Node(s_xml_dtd_declaration);
				}
			UT_CATCH(...)
				{
					default_node = 0;
				}
			if (default_node == 0)
				okay = false;

			if (okay)
				okay = m_tree->appendNode (default_node);

			UT_Element * element = 0;

			if (okay)
				{
					UT_TRY
						{
							element = new UT_Element("preferences","");
						}
					UT_CATCH(...)
						{
							element = 0;
						}
					if (element == 0)
						okay = false;
				}
			if (okay)
				okay = element->insAttr ("name", "HanAbi");
			if (okay)
				okay = element->insAttr ("version", "0.0.1");
			if (okay)
				okay = element->insAttr ("xmlns", "http://www.abisource.com/2003/HanAbi/preferences/");
			if (okay)
				okay = m_tree->appendNode (element);
			else if (element)
				delete element;
		}
	else
		{
			okay = m_tree->parse (reinterpret_cast<const char *>([data bytes]), static_cast<UT_uint32>([data length]));

			const UT_Element * root = 0;

			const UT_UTF8String * name = 0;
			const UT_UTF8String * version = 0;
			const UT_UTF8String * xmlns = 0;

			if (okay)
				{
					root = m_tree->root ();

					okay = (root->ElementTag () == "preferences");
				}
			if (okay)
				{
					name    = root->Attr ("name");
					version = root->Attr ("version");
					xmlns   = root->Attr ("xmlns");

					okay = (name && version && xmlns);
				}
			if (okay)
				{
					okay = ((*name == "HanAbi") && (*xmlns == "http://www.abisource.com/2003/HanAbi/preferences/"));
				}
		}
	if (!okay)
		{
			delete m_tree;
			m_tree = 0;
			return NO;
		}

	/* some validation/correction of basic structure; unexpected elements, and other nodes, are ignored

	<preferences name="HanAbi" version="0.0.1" xmlns="http://www.abisource.com/2003/HanAbi/preferences/">
	 <application>
	  <options>
	   ...
	  </options>
	 </application>
	 <plugins>
	  ...
	 </plugins>
	 <toolbars>
	  ...
	 </toolbars>
	</preferences>

	 */

	const UT_Element * preferences = m_tree->root ();

	const UT_Element * application = 0;
	const UT_Element * plugins = 0;
	const UT_Element * toolbars = 0;

	UT_uint32 children = preferences->children ();
	UT_uint32 i;

	for (i = 0; i < children; i++)
		{
			const UT_Element * child = preferences->child (i);

			const UT_UTF8String & tag = child->ElementTag ();

			if (tag == "application")
				{
					if (!application)
						application = child;
				}
			else if (tag == "plugins")
				{
					if (!plugins)
						plugins = child;
				}
			else if (tag == "toolbars")
				{
					if (!toolbars)
						toolbars = child;
				}
		}
	if (!(application && plugins && toolbars))
		{
			if (!application)
				{
					UT_Element * element = 0;

					UT_TRY
						{
							element = new UT_Element("application","");
						}
					UT_CATCH(...)
						{
							element = 0;
						}
					if (element == 0)
						okay = false;

					if (okay)
						okay = const_cast<UT_Element *>(preferences)->insertNode (element, 0);
					if (okay)
						application = element;
				}
			if (okay && !plugins)
				{
					UT_Element * element = 0;

					UT_TRY
						{
							element = new UT_Element("plugins","");
						}
					UT_CATCH(...)
						{
							element = 0;
						}
					if (element == 0)
						okay = false;

					if (okay)
						okay = const_cast<UT_Element *>(preferences)->appendNode (element);
					if (okay)
						plugins = element;
				}
			if (okay && !toolbars)
				{
					UT_Element * element = 0;

					UT_TRY
						{
							element = new UT_Element("toolbars","");
						}
					UT_CATCH(...)
						{
							element = 0;
						}
					if (element == 0)
						okay = false;

					if (okay)
						okay = const_cast<UT_Element *>(preferences)->appendNode (element);
					if (okay)
						toolbars = element;
				}
		}
	if (okay)
		{
			UT_uint32 nodes = preferences->nodes ();

			UT_uint32 index_plugins  = 0;
			UT_uint32 index_toolbars = 0;

			for (i = 0; i < nodes; i++)
				{
					const UT_Node * node = (*preferences)[i];

					if (node == plugins)
						index_plugins = i;
					if (node == toolbars)
						index_toolbars = i;
				}
			if (index_plugins < index_toolbars)
				okay = const_cast<UT_Element *>(preferences)->shiftNode (index_toolbars,
																		 const_cast<UT_Element *>(preferences),
																		 index_plugins + 1);
			else
				okay = const_cast<UT_Element *>(preferences)->shiftNode (index_plugins,
																		 const_cast<UT_Element *>(preferences),
																		 index_toolbars);
		}
	if (okay)
		{
			const UT_Element * options = 0;

			children = application->children ();

			for (i = 0; i < children; i++)
				{
					const UT_Element * child = application->child (i);

					if (child->ElementTag () == "options")
						{
							options = child;
							break;
						}
				}
			if (!options)
				{
					UT_Element * element = 0;

					UT_TRY
						{
							element = new UT_Element("options","");
						}
					UT_CATCH(...)
						{
							element = 0;
						}
					if (element == 0)
						okay = false;

					if (okay)
						okay = const_cast<UT_Element *>(application)->insertNode (element, 0);
					if (okay)
						options = element;
				}
		}
	if (!okay)
		{
			delete m_tree;
			m_tree = 0;
			return NO;
		}

	m_prefs = application;

	return YES;
}

- (BOOL)loadSystemPrefs
{
	UT_DEBUGMSG(("- (BOOL)loadSystemPrefs\n"));
	if (m_tree)
		{
			UT_DEBUGMSG(("can't reload system preferences file, sorry\n"));
			return NO;
		}

	NSString * prefs = @"/Library/Application Support/AbiSuite/HanAbi/HanAbi.conf";

	if ([self loadPrefsFromFile:prefs generateIfNone:NO] == NO)
		{
			UT_DEBUGMSG(("failed to load system prefs from file\n"));
			return NO;
		}

	m_system = YES;

	return YES;
}

- (BOOL)loadUserPrefs
{
	UT_DEBUGMSG(("- (BOOL)loadUserPrefs\n"));
	m_system = NO;

	if (m_tree)
		{
			UT_DEBUGMSG(("reload of user preferences file not yet implemented, sorry\n"));
			// TODO: reload prefs from file... (i.e., revert to saved)
			return NO;
		}

	NSString * prefs = @"Library/Application Support/AbiSuite/HanAbi/HanAbi.conf";

	BOOL okay = NO;

	int fd = open (".", O_RDONLY, 0);

	const char * home = getenv ("HOME");
	if (home)
		if (chdir (home) == 0)
			{
				okay = [self loadPrefsFromFile:prefs generateIfNone:YES];
				if (!okay)
					{
						UT_DEBUGMSG(("failed to load user prefs from file\n"));
					}
				fchdir (fd);
			}

	close (fd);

	return okay;
}

class HanAbiPrefsWriter : public UT_Node::Writer
{
private:
	NSMutableData *	m_data;

	UT_uint32		m_depth;

	UT_UTF8String	m_ws;

	bool			m_start;

public:
	HanAbiPrefsWriter (NSMutableData * data) :
		m_data(data),
		m_depth(0),
		m_ws("\n"),
		m_start(true)
	{
		// 
	}

	void calcIndent ()
	{
		m_ws = "\n";

		UT_uint32 depth = m_depth;

		while (depth > 8)
			{
				m_ws += "\t";
				depth -= 8;
			}
		while (depth)
			{
				m_ws += " ";
				--depth;
			}
	}

	/* if this returns false the current node (elements call the next method) isn't written
	 */
	bool treeNodeWrite (const UT_Node * node)
	{
		return (node->type () == UT_Node::nt_default);
	}

	/* in the case of elements, all descendant nodes are affected also if you return
	 * false and set descend to false
	 */
	bool treeNodeWrite (const UT_Element * element, bool & descend)
	{
		if (m_start)
			m_start = false;
		else
			treeNodeWrite (m_ws);

		++m_depth;
		calcIndent ();

		return true;
	}

	/* notify that element is about to write the end-tag
	 */
	void treeAscending (const UT_Element * element)
	{
		if (m_depth)
			--m_depth;
		calcIndent ();
	}

	/* if this returns false the write process is stopped
	 */
	bool treeNodeWrite (const UT_UTF8String & text)
	{
		[m_data appendBytes:(text.utf8_str ()) length:static_cast<unsigned>(text.byteLength ())];
		return true;
	}
};

- (BOOL)savePrefsToFile:(NSString *)filename
{
	UT_DEBUGMSG(("- (BOOL)savePrefsToFile\n"));
	if (!m_tree || !filename)
		return NO;
	if (m_parent)
		return [m_parent savePrefsToFile:filename];

	NSMutableData * data = [NSMutableData dataWithCapacity:1024];

	HanAbiPrefsWriter writer (data);
	if (!m_tree->write (writer))
		{
			UT_DEBUGMSG(("error why constructing output from preferences tree\n"));
			return NO;
		}
	return [data writeToFile:filename atomically:YES];
}

- (BOOL)saveUserPrefs
{
	UT_DEBUGMSG(("- (BOOL)saveUserPrefs\n"));
	if (!m_tree)
		return NO;
	if (m_parent)
		return [m_parent saveUserPrefs];

	NSString * prefs = @"Library/Application Support/AbiSuite/HanAbi/HanAbi.conf";

	BOOL okay = NO;

	int fd = open (".", O_RDONLY, 0);

	const char * home = getenv ("HOME");
	if (home)
		if (chdir (home) == 0)
			{
				okay = [self savePrefsToFile:prefs];
				if (!okay)
					{
						UT_DEBUGMSG(("failed to save user preferences file\n"));
					}
				fchdir (fd);
			}

	close (fd);

	return okay;
}

@end
