/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiWord / AbiBits / HanAbi
 *
 * Copyright (C) 2003 AbiSource, Inc.
 * Copyright (C) 2003 Francis James Franklin <fjf@alinameridon.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <unistd.h>
#include <fcntl.h>

#import "HanAbiPluginManager.h"
#import "HanAbiPluginHandler.h"
#import "HanAbiPluginsPanel.h"

@implementation HanAbiPluginManager

static NSString * HanAbiPluginSuffix = @".HanAbi";

static HanAbiPluginManager * defaultPluginManager = nil;

+ (HanAbiPluginManager *) defaultManager
{
	if (defaultPluginManager == nil)
		{
			defaultPluginManager = (HanAbiPluginManager *) [[HanAbiPluginManager alloc] init];
		}
	return defaultPluginManager;
}

- (id)init
{
	self = [super init];

	m_PluginDictionary = (NSMutableDictionary *) [[NSMutableDictionary alloc] initWithCapacity:16];

	return self;
}

- (void)dealloc
{
	[m_PluginDictionary release];

	[super dealloc];
}

- (NSString *)pluginSuffix
{
	return HanAbiPluginSuffix;
}

- (void)queryPlugins
{
	NSString * plugindir = @"Library/Application Support/AbiSuite/HanAbi/Plug-ins";

	/* look for plugins (*.HanAbi) in 4 places, in this order:
	 * 
	 * 1. the application's built-in plugins folder;
	 * 2. $HOME/Library/Application Support/AbiSuite/HanAbi/Plug-ins;
	 * 3. /Library/Application Support/AbiSuite/HanAbi/Plug-ins.
	 */

	int fd = open (".", O_RDONLY, 0);

	/* 1. the application's built-in plugins folder
	 */
	NSString * path = [[NSBundle mainBundle] builtInPlugInsPath];

	[self queryPluginsInDirectory:path];

	/* 2. $HOME/Library/Application Support/AbiSuite/HanAbi/Plug-ins;
	 */
	const char * home = getenv ("HOME");
	if (home)
		if (chdir (home) == 0)
			{
				[self queryPluginsInDirectory:plugindir];
			}

	/* 3. /Library/Application Support/AbiSuite/HanAbi/Plug-ins.
	 */
	if (chdir ("/") == 0)
		{
			[self queryPluginsInDirectory:plugindir];
		}

	fchdir (fd);
	close (fd);
}

- (void)queryPluginsInDirectory:(NSString *)directory
{
	NSFileManager * FM = [NSFileManager defaultManager];

	if ([FM changeCurrentDirectoryPath:directory])
		{
			NSString * cwd = [FM currentDirectoryPath];
			if (cwd)
				{
					NSArray * contents = [FM directoryContentsAtPath:@"."];
					if (contents)
						{
							unsigned count = [contents count];
							for (unsigned i = 0; i < count; i++)
								{
									NSString * filename = (NSString *) [contents objectAtIndex:i];
									if ([filename hasSuffix:HanAbiPluginSuffix])
										{
											[self queryPlugin:[cwd stringByAppendingPathComponent:filename]];
										}
								}
						}
				}
		}
}

- (void)queryPlugin:(NSString *)plugin
{
	if ([plugin hasSuffix:HanAbiPluginSuffix] == NO)
		return;

	if ([m_PluginDictionary objectForKey:plugin]) // already queried successfully
		return;

	NSFileManager * FM = [NSFileManager defaultManager];

	BOOL isDirectory = NO;

	if ([FM fileExistsAtPath:plugin isDirectory:&isDirectory] == NO)
		return;
	if (isDirectory == NO)
		return;

	NSString * configFile = [plugin stringByAppendingPathComponent:@"Contents/Resources/plugin.conf"];

	if ([FM fileExistsAtPath:configFile isDirectory:&isDirectory] == NO)
		return;
	if (isDirectory == YES)
		return;

	HanAbiPluginHandler * handler = (HanAbiPluginHandler *) [[HanAbiPluginHandler alloc] initWithBundlePath:plugin];
	if (handler)
		{
			if ([handler readPluginConf:configFile])
				{
					[m_PluginDictionary setObject:handler forKey:plugin];
				}
			[handler release];
		}
}

- (BOOL)loadPlugin:(NSString *)plugin
{
	[self queryPlugin:plugin];

	HanAbiPluginHandler * handler = (HanAbiPluginHandler *) [m_PluginDictionary objectForKey:plugin];
	if (handler == nil)
		return NO;

	if ([handler load] == NO)
		return NO;

	HanAbiPluginsPanel * panel = [HanAbiPluginsPanel sharedPluginsPanel];
	if (panel)
		[panel reloadAndSelectHandler:handler];

	return YES;
}

- (NSArray *)handlers
{
	return m_PluginDictionary ? [m_PluginDictionary allValues] : nil;
}

@end
