/* -*- mode: C++; tab-width: 4; c-basic-offset: 4; -*- */

/* AbiWord / AbiBits / HanAbi
 *
 * Copyright (C) 2003 AbiSource, Inc.
 * Copyright (C) 2003 Francis James Franklin <fjf@alinameridon.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "ut_exception.h"
#include "ut_Tree.h"

#import "HanAbiOptions.h"
#import "HanAbiPlugin.h"
#import "HanAbiPluginHandler.h"

@implementation HanAbiPluginHandler

- (id)initWithBundlePath:(NSString *)path;
{
	if (self = [super init])
		{
			m_name = nil;
			m_version = nil;
			m_description = nil;

			m_path = path;
			[m_path retain];

			m_bundle = nil;
			m_instance = nil;

			UT_TRY
				{
					m_config = new UT_Tree;
				}
			UT_CATCH(...)
				{
					m_config = 0;
				}
			if (m_config == 0)
				{
					[self release];
					return nil;
				}
		}
	return self;
}

- (void)dealloc
{
	if (m_name)
		[m_name release];
	if (m_version)
		[m_version release];
	if (m_description)
		[m_description release];
	if (m_path)
		[m_path release];

	if (m_instance)
		[m_instance release];

	if (m_bundle)
		[m_bundle release];

	if (m_config)
		delete m_config;
}

- (NSString *)name
{
	return m_name;
}

- (NSString *)version
{
	return m_version;
}

- (NSString *)description
{
	return m_description;
}

- (BOOL)readPluginConf:(NSString *)filename
{
	NSData * data = (NSData *) [NSData dataWithContentsOfFile:filename];

	if ([data length] == 0)
		return NO;

	bool okay = m_config->parse (reinterpret_cast<const char *>([data bytes]), static_cast<UT_uint32>([data length]));

	const UT_Element * root = 0;

	const UT_UTF8String * name = 0;
	const UT_UTF8String * version = 0;
	const UT_UTF8String * xmlns = 0;

	if (okay)
		{
			root = m_config->root ();

			okay = (root->ElementTag () == "plugin");
		}
	if (okay)
		{
			name    = root->Attr ("name");
			version = root->Attr ("version");
			xmlns   = root->Attr ("xmlns");

			okay = (name && version && xmlns);
		}
	if (okay)
		{
			okay = (name->byteLength () && (*xmlns == "http://www.abisource.com/2003/HanAbi/plugin-conf/"));
		}
	if (okay)
		{
			m_name    = (NSString *) [[NSString alloc] initWithUTF8String:(name->utf8_str ())];
			m_version = (NSString *) [[NSString alloc] initWithUTF8String:(version->utf8_str ())];

			[self composeDescription];
		}
	return (okay ? YES : NO);
}

- (void)composeDescription
{
	if (m_description)
		return;

	const UT_Element * plugin = m_config->root ();
	const UT_Element * authors = 0;
	const UT_Element * description = 0;

	UT_uint32 children = plugin->children ();

	NSString * str = @"";

	UT_uint32 i;

	for (i = 0; i < children; i++)
		{
			if (plugin->child(i)->ElementTag () == "authors")
				authors = plugin->child (i);
			if (plugin->child(i)->ElementTag () == "description")
				description = plugin->child (i);
		}
	if (description)
		{
			UT_uint32 nodes = description->nodes ();

			for (UT_uint32 n = 0; n < nodes; n++)
				{
					const UT_Node * node = (*description)[n];
					if ((node->type () == UT_Node::nt_text) || (node->type () == UT_Node::nt_cdata))
						{
							const UT_Text_Node * text_node = static_cast<const UT_Text_Node *>(node);
							NSString * text = [NSString stringWithUTF8String:(text_node->text().utf8_str ())];
							str = [str stringByAppendingString:text];
						}
				}
		}
	if (authors)
		{
			children = authors->children ();

			for (i = 0; i < children; i++)
				{
					const UT_Element * author = authors->child (i);
					const UT_UTF8String * copyright = author->Attr ("copyright");

					NSString * copy = @"\n";
					if (copyright)
						if (copyright->byteLength ())
							{
								NSString * text = [NSString stringWithUTF8String:(copyright->utf8_str ())];

								copy = @"\nCopyright (C) ";
								copy = [copy stringByAppendingString:text];
								copy = [copy stringByAppendingString:@" "];
							}

					UT_uint32 nodes = author->nodes ();

					for (UT_uint32 n = 0; n < nodes; n++)
						{
							const UT_Node * node = (*author)[n];
							if ((node->type () == UT_Node::nt_text) || (node->type () == UT_Node::nt_cdata))
								{
									const UT_Text_Node * text_node = static_cast<const UT_Text_Node *>(node);
									NSString * text = [NSString stringWithUTF8String:(text_node->text().utf8_str ())];
									copy = [copy stringByAppendingString:text];
								}
						}
					str = [str stringByAppendingString:copy];
				}
		}
	m_description = str;

	[m_description retain];
}

- (const UT_Tree *)pluginConf
{
	return m_config;
}

- (NSString *)valueForOption:(NSString *)key
{
	if (!m_config)
		return nil;

	const UT_Element * plugin = m_config->root ();

	if (!plugin)
		return nil;

	const char * value = HanAbi_GetOption (plugin, [key UTF8String]);

	return value ? [NSString stringWithUTF8String:value] : nil;
}

- (BOOL)isLoaded
{
	return (m_bundle ? [m_bundle isLoaded] : NO);
}

- (BOOL)load
{
	if (m_bundle)
		return [m_bundle isLoaded];

	m_bundle = (NSBundle *) [[NSBundle alloc] initWithPath:m_path];
	if (m_bundle == nil)
		return NO;

	return [m_bundle load];
}

- (BOOL)isActive
{
	return (m_instance ? YES : NO);
}

- (BOOL)activate
{
	if ([self isLoaded] == NO)
		if ([self load] == NO)
			return NO; // TODO: popup alert box?

	if (m_instance)
		return YES;

	Class bundleClass = [m_bundle principalClass];
	if (bundleClass == nil)
		return NO;

	m_instance = (HanAbiPlugin *) [[bundleClass alloc] initWithHandler:self];

	return (m_instance ? YES : NO);
}

- (void)deactivate
{
	if (m_instance)
		{
			[m_instance release];
			m_instance  = nil;
		}
}

- (void)displayOptions
{
	if (m_instance)
		[m_instance displayOptions];
}

@end
